"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useWithManagedConsoleState = exports.useWithManagedConsole = exports.useConsoleManager = exports.ConsoleManager = void 0;
var _extends2 = _interopRequireDefault(require("@babel/runtime/helpers/extends"));
var _react = _interopRequireWildcard(require("react"));
var _console_page_overlay = require("./components/console_page_overlay");
var _console = require("../../console");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ConsoleManagerContext = /*#__PURE__*/_react.default.createContext(undefined);
/**
 * A console management context. Allow for the show/hide of consoles without them loosing their
 * command history while running in "hidden" mode.
 */
const ConsoleManager = exports.ConsoleManager = /*#__PURE__*/(0, _react.memo)(({
  storage = {},
  children
}) => {
  const [consoleStorage, setConsoleStorage] = (0, _react.useState)(storage);
  const [consoleStateStorage] = (0, _react.useState)(new Map());

  // `consoleStorageRef` keeps a copy (reference) to the latest copy of the `consoleStorage` so that
  // some exposed methods (ex. `RegisteredConsoleClient`) are guaranteed to be immutable and function
  // as expected between state updates without having to re-update every record stored in the `ConsoleStorage`
  const consoleStorageRef = (0, _react.useRef)();
  consoleStorageRef.current = consoleStorage;
  const validateIdOrThrow = (0, _react.useCallback)(id => {
    var _consoleStorageRef$cu;
    if (!((_consoleStorageRef$cu = consoleStorageRef.current) !== null && _consoleStorageRef$cu !== void 0 && _consoleStorageRef$cu[id])) {
      throw new Error(`Console with id ${id} not found`);
    }
  }, []); // << IMPORTANT: this callback should have no dependencies

  const show = (0, _react.useCallback)(id => {
    validateIdOrThrow(id);
    setConsoleStorage(prevState => {
      const newState = {
        ...prevState
      };

      // if any is visible, hide it
      Object.entries(newState).forEach(([consoleId, managedConsole]) => {
        if (managedConsole.isOpen) {
          newState[consoleId] = {
            ...managedConsole,
            isOpen: false
          };
        }
      });
      newState[id] = {
        ...newState[id],
        isOpen: true
      };
      return newState;
    });
  }, [validateIdOrThrow] // << IMPORTANT: this callback should have only immutable dependencies
  );
  const hide = (0, _react.useCallback)(id => {
    validateIdOrThrow(id);
    setConsoleStorage(prevState => {
      return {
        ...prevState,
        [id]: {
          ...prevState[id],
          isOpen: false
        }
      };
    });
  }, [validateIdOrThrow] // << IMPORTANT: this callback should have only immutable dependencies
  );
  const terminate = (0, _react.useCallback)(id => {
    validateIdOrThrow(id);
    setConsoleStorage(prevState => {
      const newState = {
        ...prevState
      };
      delete newState[id];
      return newState;
    });
  }, [validateIdOrThrow] // << IMPORTANT: this callback should have only immutable dependencies
  );
  const getOne = (0, _react.useCallback)(id => {
    var _consoleStorageRef$cu2;
    if ((_consoleStorageRef$cu2 = consoleStorageRef.current) !== null && _consoleStorageRef$cu2 !== void 0 && _consoleStorageRef$cu2[id]) {
      return consoleStorageRef.current[id].client;
    }
  }, [] // << IMPORTANT: this callback should have no dependencies or only immutable dependencies
  );
  const getList = (0, _react.useCallback)(() => {
    return Object.values(consoleStorage).map(managedConsole => managedConsole.client);
  }, [consoleStorage]); // << This callback should always use `consoleStorage`

  const isVisible = (0, _react.useCallback)(id => {
    var _consoleStorageRef$cu3;
    if ((_consoleStorageRef$cu3 = consoleStorageRef.current) !== null && _consoleStorageRef$cu3 !== void 0 && _consoleStorageRef$cu3[id]) {
      return consoleStorageRef.current[id].isOpen;
    }
    return false;
  }, []); // << IMPORTANT: this callback should have no dependencies

  const register = (0, _react.useCallback)(({
    id,
    meta,
    consoleProps,
    ...otherRegisterProps
  }) => {
    if (consoleStorage[id]) {
      throw new Error(`Console with id ${id} already registered`);
    }
    const managedKey = Symbol(id);
    // Referencing/using the interface methods here (defined in the outer scope of this function)
    // is ok because those are immutable and thus will not change between state changes
    const showThisConsole = show.bind(null, id);
    const hideThisConsole = hide.bind(null, id);
    const terminateThisConsole = terminate.bind(null, id);
    const isThisConsoleVisible = isVisible.bind(null, id);
    const managedConsole = {
      PageBodyComponent: undefined,
      PageTitleComponent: undefined,
      ActionComponents: undefined,
      ...otherRegisterProps,
      client: {
        id,
        meta,
        // The use of `setTimeout()` below is needed because this client interface can be consumed
        // prior to the component state being updated. Placing a delay on the execution of these
        // methods allows for state to be updated first and then the action is applied.
        // So someone can do: `.register({...}).show()` and it will work
        show: () => {
          setTimeout(showThisConsole, 0);
        },
        hide: () => {
          setTimeout(hideThisConsole, 0);
        },
        terminate: () => {
          setTimeout(terminateThisConsole, 0);
        },
        isVisible: () => isThisConsoleVisible()
      },
      consoleProps,
      console: /*#__PURE__*/_react.default.createElement(_console.Console, (0, _extends2.default)({}, consoleProps, {
        managedKey: managedKey,
        key: id
      })),
      isOpen: false,
      key: managedKey
    };
    setConsoleStorage(prevState => {
      return {
        ...prevState,
        [id]: managedConsole
      };
    });
    return managedConsole.client;
  }, [consoleStorage, hide, isVisible, show, terminate]);
  const consoleManagerClient = (0, _react.useMemo)(() => {
    return {
      register,
      show,
      hide,
      terminate,
      getOne,
      getList
    };
  }, [getList, getOne, hide, register, show, terminate]);
  const consoleManageContextClients = (0, _react.useMemo)(() => {
    return {
      client: consoleManagerClient,
      internal: {
        getManagedConsole(key) {
          return Object.values(consoleStorage).find(managedConsole => managedConsole.key === key);
        },
        getManagedConsoleState(key) {
          return consoleStateStorage.get(key);
        },
        storeManagedConsoleState(key, state) {
          consoleStateStorage.set(key, state);
        }
      }
    };
  }, [consoleManagerClient, consoleStateStorage, consoleStorage]);
  const visibleConsole = (0, _react.useMemo)(() => {
    return Object.values(consoleStorage).find(managedConsole => managedConsole.isOpen);
  }, [consoleStorage]);
  const visibleConsoleMeta = (0, _react.useMemo)(() => {
    var _visibleConsole$clien;
    return (_visibleConsole$clien = visibleConsole === null || visibleConsole === void 0 ? void 0 : visibleConsole.client.meta) !== null && _visibleConsole$clien !== void 0 ? _visibleConsole$clien : {};
  }, [visibleConsole === null || visibleConsole === void 0 ? void 0 : visibleConsole.client.meta]);
  const handleOnHide = (0, _react.useCallback)(() => {
    if (visibleConsole) {
      consoleManagerClient.hide(visibleConsole.client.id);
    }
  }, [consoleManagerClient, visibleConsole]);
  return /*#__PURE__*/_react.default.createElement(ConsoleManagerContext.Provider, {
    value: consoleManageContextClients
  }, children, visibleConsole && /*#__PURE__*/_react.default.createElement(_console_page_overlay.ConsolePageOverlay, {
    onHide: handleOnHide,
    console: /*#__PURE__*/_react.default.createElement(_console.Console, (0, _extends2.default)({}, visibleConsole.consoleProps, {
      managedKey: visibleConsole.key,
      key: visibleConsole.client.id
    })),
    isHidden: !visibleConsole,
    pageTitle: visibleConsole.PageTitleComponent && /*#__PURE__*/_react.default.createElement(visibleConsole.PageTitleComponent, {
      meta: visibleConsoleMeta
    }),
    body: visibleConsole.PageBodyComponent && /*#__PURE__*/_react.default.createElement(visibleConsole.PageBodyComponent, {
      meta: visibleConsoleMeta
    }),
    actions: visibleConsole.ActionComponents && visibleConsole.ActionComponents.map(ActionComponent => {
      return /*#__PURE__*/_react.default.createElement(ActionComponent, {
        meta: visibleConsoleMeta
      });
    }),
    showCloseButton: visibleConsole.showCloseButton
  }));
});
ConsoleManager.displayName = 'ConsoleManager';

/**
 * Returns the interface for managing consoles withing a `<ConsoleManager/>` context.
 */
const useConsoleManager = () => {
  const consoleManagerClients = (0, _react.useContext)(ConsoleManagerContext);
  if (!consoleManagerClients) {
    throw new Error('ConsoleManagerContext not found');
  }
  return consoleManagerClients.client;
};

/**
 * For internal use within Console code only!
 * Hook will return the `ManagedConsole` interface stored in the manager if it finds
 * the `ConsoleProps` provided on input to be one that the ConsoleManager is tracking.
 *
 * @protected
 */
exports.useConsoleManager = useConsoleManager;
const useWithManagedConsole = key => {
  const consoleManagerClients = (0, _react.useContext)(ConsoleManagerContext);
  if (key && consoleManagerClients) {
    return consoleManagerClients.internal.getManagedConsole(key);
  }
};
exports.useWithManagedConsole = useWithManagedConsole;
/**
 * Provides methods for retrieving/storing a console's internal state (if any)
 * @param key
 */
const useWithManagedConsoleState = key => {
  const consoleManagerClients = (0, _react.useContext)(ConsoleManagerContext);
  return (0, _react.useMemo)(() => {
    if (!key || !consoleManagerClients) {
      return [undefined, undefined];
    }
    return [
    // getState()
    () => {
      return consoleManagerClients.internal.getManagedConsoleState(key);
    },
    // storeState()
    state => {
      if (consoleManagerClients.internal.getManagedConsole(key)) {
        consoleManagerClients.internal.storeManagedConsoleState(key, state);
      }
    }];
  }, [consoleManagerClients, key]);
};
exports.useWithManagedConsoleState = useWithManagedConsoleState;