"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useNavigationSteps = exports.useFileValidation = void 0;
var _i18n = require("@kbn/i18n");
var _papaparse = require("papaparse");
var _react = require("react");
var _fp = require("lodash/fp");
var _formatted_bytes = require("../../../common/components/formatted_bytes");
var _validations = require("./validations");
var _kibana = require("../../../common/lib/kibana");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useFileValidation = ({
  onError,
  onComplete
}) => {
  const formatBytes = (0, _formatted_bytes.useFormatBytes)();
  const {
    telemetry
  } = (0, _kibana.useKibana)().services;
  const onErrorWrapper = (0, _react.useCallback)((error, file) => {
    telemetry.reportAssetCriticalityFileSelected({
      valid: false,
      errorCode: error.code,
      file: {
        size: file.size
      }
    });
    onError(error.message, file);
  }, [onError, telemetry]);
  return (0, _react.useCallback)(file => {
    const processingStartTime = Date.now();
    const fileValidation = (0, _validations.validateFile)(file, formatBytes);
    if (!fileValidation.valid) {
      onErrorWrapper({
        message: fileValidation.errorMessage,
        code: fileValidation.code
      }, file);
      return;
    }
    telemetry.reportAssetCriticalityFileSelected({
      valid: true,
      file: {
        size: file.size
      }
    });
    const parserConfig = {
      dynamicTyping: true,
      skipEmptyLines: true,
      complete(parsedFile, returnedFile) {
        var _returnedFile$name, _returnedFile$size;
        if (parsedFile.data.length === 0) {
          onErrorWrapper({
            message: _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.assetCriticalityFileUploader.emptyFileError', {
              defaultMessage: 'The file is empty'
            })
          }, file);
          return;
        }
        const {
          invalid,
          valid,
          errors
        } = (0, _validations.validateParsedContent)(parsedFile.data);
        const validLinesAsText = (0, _papaparse.unparse)(valid);
        const invalidLinesAsText = (0, _papaparse.unparse)(invalid);
        const processingEndTime = Date.now();
        const tookMs = processingEndTime - processingStartTime;
        onComplete({
          processingStartTime: new Date(processingStartTime).toISOString(),
          processingEndTime: new Date(processingEndTime).toISOString(),
          tookMs,
          validatedFile: {
            name: (_returnedFile$name = returnedFile === null || returnedFile === void 0 ? void 0 : returnedFile.name) !== null && _returnedFile$name !== void 0 ? _returnedFile$name : '',
            size: (_returnedFile$size = returnedFile === null || returnedFile === void 0 ? void 0 : returnedFile.size) !== null && _returnedFile$size !== void 0 ? _returnedFile$size : 0,
            validLines: {
              text: validLinesAsText,
              count: valid.length
            },
            invalidLines: {
              text: invalidLinesAsText,
              count: invalid.length,
              errors
            }
          }
        });
      },
      error(parserError) {
        onErrorWrapper({
          message: parserError.message
        }, file);
      }
    };
    (0, _papaparse.parse)(file, parserConfig);
  }, [formatBytes, telemetry, onErrorWrapper, onComplete]);
};
exports.useFileValidation = useFileValidation;
const useNavigationSteps = (state, goToFirstStep) => {
  return (0, _react.useMemo)(() => [{
    title: _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.assetCriticalityUploadPage.selectFileStepTitle', {
      defaultMessage: 'Select a file'
    }),
    status: (0, _helpers.getStepStatus)(1, state.step),
    onClick: () => {
      if ((0, _helpers.isValidationStep)(state)) {
        goToFirstStep(); // User can only go back to the first step from the second step
      }
    }
  }, {
    title: _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.assetCriticalityUploadPage.fileValidationStepTitle', {
      defaultMessage: 'File validation'
    }),
    status: (0, _helpers.getStepStatus)(2, state.step),
    onClick: _fp.noop // Prevents the user from navigating by clicking on the step
  }, {
    title: _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.assetCriticalityUploadPage.resultsStepTitle', {
      defaultMessage: 'Results'
    }),
    status: (0, _helpers.getStepStatus)(3, state.step),
    onClick: _fp.noop // Prevents the user from navigating by clicking on the step
  }], [goToFirstStep, state]);
};
exports.useNavigationSteps = useNavigationSteps;