"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAssetCriticalityPrivileges = exports.useAssetCriticalityData = void 0;
var _reactQuery = require("@tanstack/react-query");
var _public = require("@kbn/kibana-react-plugin/public");
var _constants = require("../../../../common/constants");
var _helper_hooks = require("../../../helper_hooks");
var _api = require("../../api/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ASSET_CRITICALITY_KEY = 'ASSET_CRITICALITY';
const PRIVILEGES_KEY = 'PRIVILEGES';
const nonAuthorizedResponse = Promise.resolve({
  has_all_required: false,
  has_write_permissions: false,
  has_read_permissions: false,
  privileges: {
    elasticsearch: {}
  }
});
const useAssetCriticalityPrivileges = queryKey => {
  const {
    fetchAssetCriticalityPrivileges
  } = (0, _api.useEntityAnalyticsRoutes)();
  const hasEntityAnalyticsCapability = (0, _helper_hooks.useHasSecurityCapability)('entity-analytics');
  const [isAssetCriticalityEnabled] = (0, _public.useUiSetting$)(_constants.ENABLE_ASSET_CRITICALITY_SETTING);
  const isEnabled = isAssetCriticalityEnabled && hasEntityAnalyticsCapability;
  return (0, _reactQuery.useQuery)({
    queryKey: [ASSET_CRITICALITY_KEY, PRIVILEGES_KEY, queryKey, isEnabled],
    queryFn: isEnabled ? fetchAssetCriticalityPrivileges : () => nonAuthorizedResponse
  });
};
exports.useAssetCriticalityPrivileges = useAssetCriticalityPrivileges;
const useAssetCriticalityData = ({
  entity,
  enabled = true
}) => {
  const QC = (0, _reactQuery.useQueryClient)();
  const QUERY_KEY = [ASSET_CRITICALITY_KEY, entity.name];
  const {
    fetchAssetCriticality,
    createAssetCriticality,
    deleteAssetCriticality
  } = (0, _api.useEntityAnalyticsRoutes)();
  const privileges = useAssetCriticalityPrivileges(entity.name);
  const query = (0, _reactQuery.useQuery)({
    queryKey: QUERY_KEY,
    queryFn: () => fetchAssetCriticality({
      idField: `${entity.type}.name`,
      idValue: entity.name
    }),
    retry: (failureCount, error) => error.body.statusCode === 404 && failureCount > 0,
    enabled
  });
  const mutation = (0, _reactQuery.useMutation)({
    mutationFn: params => {
      if (params.criticalityLevel === 'unassigned') {
        return deleteAssetCriticality({
          idField: params.idField,
          idValue: params.idValue
        });
      }
      return createAssetCriticality({
        idField: params.idField,
        idValue: params.idValue,
        criticalityLevel: params.criticalityLevel
      });
    },
    onSuccess: data => {
      const queryData = 'deleted' in data ? null : data;
      QC.setQueryData(QUERY_KEY, queryData);
    }
  });
  const was404 = query.isError && query.error.body.statusCode === 404;
  const returnedData = query.isSuccess && query.data != null;
  const status = was404 || !returnedData ? 'create' : 'update';
  return {
    status,
    query,
    mutation,
    privileges
  };
};
exports.useAssetCriticalityData = useAssetCriticalityData;