"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFindAppLinksByPath = void 0;
var _securitySolutionNavigation = require("@kbn/security-solution-navigation");
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _common = require("../../../common");
var _kibana = require("../lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * It returns the first nav item that matches the current pathname.
 * It compares the pathname and nav item using `startsWith`,
 * meaning that the pathname: `/hosts/anomalies` matches the nav item URL `/hosts`.
 */
const useFindAppLinksByPath = navLinks => {
  const {
    getAppUrl
  } = (0, _kibana.useAppUrl)();
  const basePath = (0, _kibana.useBasePath)();
  const {
    pathname
  } = (0, _reactRouterDom.useLocation)();
  const isCurrentPathItem = (0, _react.useCallback)(navItem => {
    const {
      appId,
      deepLinkId
    } = (0, _securitySolutionNavigation.getNavigationPropsFromId)(navItem.id);
    const appUrl = getAppUrl({
      appId,
      deepLinkId
    });
    return !!(0, _reactRouterDom.matchPath)(`${basePath}${_common.APP_PATH}${pathname}`, {
      path: appUrl,
      strict: false
    });
  }, [basePath, getAppUrl, pathname]);
  return (0, _react.useMemo)(() => findNavItem(isCurrentPathItem, navLinks), [navLinks, isCurrentPathItem]);
};

/**
 * DFS to find the first nav item that matches the current pathname.
 * Case the leaf node does not match the pathname; we return the nearest parent node that does.
 *
 * @param predicate calls predicate once for each element of the tree, until it finds one where predicate returns true.
 */
exports.useFindAppLinksByPath = useFindAppLinksByPath;
const findNavItem = (predicate, navItems) => {
  if (!navItems) return null;
  for (const navItem of navItems) {
    var _navItem$links;
    if ((_navItem$links = navItem.links) !== null && _navItem$links !== void 0 && _navItem$links.length) {
      const foundItem = findNavItem(predicate, navItem.links);
      if (foundItem) return foundItem;
    }
    if (predicate(navItem)) {
      return navItem;
    }
  }
  return null;
};