"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.singleEntryThreat = exports.getUpdatedEntriesOnDelete = exports.getNewItem = exports.getFormattedEntry = exports.getFormattedEntries = exports.getEntryOnThreatFieldChange = exports.getEntryOnFieldChange = exports.getDefaultEmptyEntry = exports.filterItems = exports.customValidators = exports.containsInvalidItems = void 0;
var _uuid = require("uuid");
var _i18n = require("@kbn/i18n");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
var _securitysolutionIoTsAlertingTypes = require("@kbn/securitysolution-io-ts-alerting-types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Formats the entry into one that is easily usable for the UI.
 *
 * @param patterns DataViewBase containing available fields on rule index
 * @param item item entry
 * @param itemIndex entry index
 */
const getFormattedEntry = (indexPattern, threatIndexPatterns, item, itemIndex, uuidGen = _uuid.v4) => {
  var _maybeId$id;
  const {
    fields
  } = indexPattern;
  const {
    fields: threatFields
  } = threatIndexPatterns;
  const field = item.field;
  const threatField = item.value;
  const [foundField] = fields.filter(({
    name
  }) => field != null && field === name);
  const [threatFoundField] = threatFields.filter(({
    name
  }) => threatField != null && threatField === name);
  const maybeId = item;
  return {
    id: (_maybeId$id = maybeId.id) !== null && _maybeId$id !== void 0 ? _maybeId$id : uuidGen(),
    field: foundField,
    type: 'mapping',
    value: threatFoundField,
    entryIndex: itemIndex
  };
};

/**
 * Formats the entries to be easily usable for the UI
 *
 * @param patterns DataViewBase containing available fields on rule index
 * @param entries item entries
 */
exports.getFormattedEntry = getFormattedEntry;
const getFormattedEntries = (indexPattern, threatIndexPatterns, entries) => {
  return entries.reduce((acc, item, index) => {
    const newItemEntry = getFormattedEntry(indexPattern, threatIndexPatterns, item, index);
    return [...acc, newItemEntry];
  }, []);
};

/**
 * Determines whether an entire entry or item need to be removed
 *
 * @param item
 * @param entryIndex index of given entry
 *
 */
exports.getFormattedEntries = getFormattedEntries;
const getUpdatedEntriesOnDelete = (item, entryIndex) => {
  return {
    ...item,
    entries: [...item.entries.slice(0, entryIndex), ...item.entries.slice(entryIndex + 1)]
  };
};

/**
 * Determines proper entry update when user selects new field
 *
 * @param item - current item entry values
 * @param newField - newly selected field
 *
 */
exports.getUpdatedEntriesOnDelete = getUpdatedEntriesOnDelete;
const getEntryOnFieldChange = (item, newField) => {
  const {
    entryIndex
  } = item;
  return {
    updatedEntry: {
      id: item.id,
      field: newField != null ? newField.name : '',
      type: 'mapping',
      value: item.value != null ? item.value.name : ''
    },
    // Cast to Entry since id is only used as a react key prop and can be ignored elsewhere
    index: entryIndex
  };
};

/**
 * Determines proper entry update when user selects new field
 *
 * @param item - current item entry values
 * @param newField - newly selected field
 *
 */
exports.getEntryOnFieldChange = getEntryOnFieldChange;
const getEntryOnThreatFieldChange = (item, newField) => {
  const {
    entryIndex
  } = item;
  return {
    updatedEntry: {
      id: item.id,
      field: item.field != null ? item.field.name : '',
      type: 'mapping',
      value: newField != null ? newField.name : ''
    },
    // Cast to Entry since id is only used as a react key prop and can be ignored elsewhere
    index: entryIndex
  };
};
exports.getEntryOnThreatFieldChange = getEntryOnThreatFieldChange;
const getDefaultEmptyEntry = () => {
  return (0, _securitysolutionUtils.addIdToItem)({
    field: '',
    type: 'mapping',
    value: ''
  });
};
exports.getDefaultEmptyEntry = getDefaultEmptyEntry;
const getNewItem = () => {
  return (0, _securitysolutionUtils.addIdToItem)({
    entries: [(0, _securitysolutionUtils.addIdToItem)({
      field: '',
      type: 'mapping',
      value: ''
    })]
  });
};
exports.getNewItem = getNewItem;
const filterItems = items => {
  return items.reduce((acc, item) => {
    const newItem = {
      ...item,
      entries: item.entries
    };
    if (_securitysolutionIoTsAlertingTypes.threatMap.is(newItem)) {
      return [...acc, newItem];
    } else {
      return acc;
    }
  }, []);
};

/**
 * Given a list of items checks each one to see if any of them have an empty field
 * or an empty value.
 * @param items The items to check if we have an empty entries.
 */
exports.filterItems = filterItems;
const containsInvalidItems = items => {
  return items.some(item => item.entries.some(subEntry => subEntry.field === '' || subEntry.value === ''));
};

/**
 * Given a list of items checks if we have a single empty entry and if we do returns true.
 * @param items The items to check if we have a single empty entry.
 */
exports.containsInvalidItems = containsInvalidItems;
const singleEntryThreat = items => {
  return items.length === 1 && items[0].entries.length === 1 && items[0].entries[0].field === '' && items[0].entries[0].value === '';
};
exports.singleEntryThreat = singleEntryThreat;
const customValidators = exports.customValidators = {
  forbiddenField: (value, forbiddenString) => {
    let match;
    if (typeof value === 'string') {
      match = value === forbiddenString;
    } else if (Array.isArray(value)) {
      match = !!value.find(item => item === forbiddenString);
    } else {
      match = false;
    }
    if (match) {
      return {
        code: 'ERR_FIELD_FORMAT',
        message: _i18n.i18n.translate('xpack.securitySolution.detectionEngine.createRule.stepDefineRule.threatMatchIndexForbiddenError', {
          defaultMessage: 'The index pattern cannot be { forbiddenString }. Please choose a more specific index pattern.',
          values: {
            forbiddenString
          }
        })
      };
    }
  }
};