"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useTogglePanel = void 0;
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _securitySolutionNavigation = require("@kbn/security-solution-navigation");
var _common = require("../../../../../../common");
var _storage = require("../storage");
var _reducer = require("../reducer");
var _types = require("../types");
var _helpers = require("../helpers");
var _configs = require("../configs");
var _kibana = require("../../../../lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const syncExpandedCardStepsToStorageFromURL = (onboardingStorage, maybeStepId) => {
  const {
    matchedCard,
    matchedStep
  } = (0, _helpers.findCardSectionByStepId)(maybeStepId);
  const hasStepContent = matchedStep && matchedStep.description;
  if (matchedCard && matchedStep && hasStepContent) {
    onboardingStorage.resetAllExpandedCardStepsToStorage();
    onboardingStorage.addExpandedCardStepToStorage(matchedCard.id, matchedStep.id);
  }
};
const syncExpandedCardStepsFromStorageToURL = (expandedCardSteps, callback) => {
  const expandedCardStep = Object.values(expandedCardSteps).find(expandedCard => expandedCard.expandedSteps.length > 0);
  if (expandedCardStep !== null && expandedCardStep !== void 0 && expandedCardStep.expandedSteps[0]) {
    const {
      matchedCard,
      matchedStep
    } = (0, _helpers.findCardSectionByStepId)(expandedCardStep === null || expandedCardStep === void 0 ? void 0 : expandedCardStep.expandedSteps[0]);
    callback === null || callback === void 0 ? void 0 : callback({
      matchedCard,
      matchedStep
    });
  }
};
const useTogglePanel = ({
  productTypes,
  onboardingSteps,
  spaceId
}) => {
  const {
    telemetry
  } = (0, _kibana.useKibana)().services;
  const {
    navigateTo
  } = (0, _securitySolutionNavigation.useNavigateTo)();
  const {
    hash: detailName
  } = (0, _reactRouterDom.useLocation)();
  const stepIdFromHash = detailName.split('#')[1];
  const onboardingStorage = (0, _react.useMemo)(() => new _storage.OnboardingStorage(spaceId), [spaceId]);
  const {
    getAllFinishedStepsFromStorage,
    getActiveProductsFromStorage,
    toggleActiveProductsInStorage,
    addExpandedCardStepToStorage,
    addFinishedStepToStorage,
    removeFinishedStepFromStorage,
    removeExpandedCardStepFromStorage,
    resetAllExpandedCardStepsToStorage,
    getAllExpandedCardStepsFromStorage
  } = onboardingStorage;
  const finishedStepsInitialStates = (0, _react.useMemo)(() => (0, _reducer.getFinishedStepsInitialStates)({
    finishedSteps: getAllFinishedStepsFromStorage()
  }), [getAllFinishedStepsFromStorage]);
  const activeProductsInitialStates = (0, _react.useMemo)(() => {
    const activeProductsFromStorage = (0, _reducer.getActiveProductsInitialStates)({
      activeProducts: getActiveProductsFromStorage()
    });
    return activeProductsFromStorage.size > 0 ? activeProductsFromStorage : productTypes && productTypes.length > 0 ? new Set(productTypes.map(({
      product_line: productLine
    }) => _configs.ProductLine[productLine])) : new Set(_configs.ALL_PRODUCT_LINES);
  }, [getActiveProductsFromStorage, productTypes]);
  const {
    activeSections: activeSectionsInitialStates,
    totalActiveSteps: totalActiveStepsInitialStates,
    totalStepsLeft: totalStepsLeftInitialStates
  } = (0, _react.useMemo)(() => (0, _reducer.getActiveSectionsInitialStates)({
    activeProducts: activeProductsInitialStates,
    finishedSteps: finishedStepsInitialStates,
    onboardingSteps
  }), [activeProductsInitialStates, finishedStepsInitialStates, onboardingSteps]);
  const expandedCardsInitialStates = (0, _react.useMemo)(() => {
    if (stepIdFromHash) {
      syncExpandedCardStepsToStorageFromURL(onboardingStorage, stepIdFromHash);
    }
    return getAllExpandedCardStepsFromStorage();
  }, [onboardingStorage, getAllExpandedCardStepsFromStorage, stepIdFromHash]);
  const [state, dispatch] = (0, _react.useReducer)(_reducer.reducer, {
    activeProducts: activeProductsInitialStates,
    activeSections: activeSectionsInitialStates,
    expandedCardSteps: expandedCardsInitialStates,
    finishedSteps: finishedStepsInitialStates,
    totalActiveSteps: totalActiveStepsInitialStates,
    totalStepsLeft: totalStepsLeftInitialStates,
    onboardingSteps
  });
  const onStepClicked = (0, _react.useCallback)(({
    stepId,
    cardId,
    isExpanded,
    trigger
  }) => {
    dispatch({
      type: _types.OnboardingActions.ToggleExpandedStep,
      payload: {
        stepId,
        cardId,
        isStepExpanded: isExpanded
      }
    });
    if (isExpanded) {
      // It allows Only One step open at a time
      resetAllExpandedCardStepsToStorage();
      addExpandedCardStepToStorage(cardId, stepId);
      telemetry.reportOnboardingHubStepOpen({
        stepId,
        trigger
      });
    } else {
      removeExpandedCardStepFromStorage(cardId, stepId);
    }
  }, [addExpandedCardStepToStorage, removeExpandedCardStepFromStorage, resetAllExpandedCardStepsToStorage, telemetry]);
  const toggleTaskCompleteStatus = (0, _react.useCallback)(({
    stepId,
    stepLinkId,
    cardId,
    sectionId,
    undo,
    trigger
  }) => {
    dispatch({
      type: undo ? _types.OnboardingActions.RemoveFinishedStep : _types.OnboardingActions.AddFinishedStep,
      payload: {
        stepId,
        cardId,
        sectionId
      }
    });
    if (undo) {
      removeFinishedStepFromStorage(cardId, stepId, state.onboardingSteps);
    } else {
      addFinishedStepToStorage(cardId, stepId);
      telemetry.reportOnboardingHubStepFinished({
        stepId,
        stepLinkId,
        trigger
      });
    }
  }, [addFinishedStepToStorage, removeFinishedStepFromStorage, state.onboardingSteps, telemetry]);
  const onProductSwitchChanged = (0, _react.useCallback)(section => {
    dispatch({
      type: _types.OnboardingActions.ToggleProduct,
      payload: {
        section: section.id
      }
    });
    toggleActiveProductsInStorage(section.id);
  }, [toggleActiveProductsInStorage]);
  (0, _react.useEffect)(() => {
    /** Handle landing on the page without hash
     ** e.g.: https://localhost:5601/app/security/get_started
     ** If there is no expanded card step in storage, do nothing.
     ** If there is expanded card step in storage, sync it to the url.
     **/
    if (!stepIdFromHash) {
      // If all steps are collapsed, do nothing
      if (Object.values(state.expandedCardSteps).every(c => !c.isExpanded)) {
        return;
      }
      syncExpandedCardStepsFromStorageToURL(expandedCardsInitialStates, ({
        matchedStep
      }) => {
        if (!matchedStep) return;
        navigateTo({
          deepLinkId: _common.SecurityPageName.landing,
          path: `#${matchedStep.id}`
        });
      });
    }
  }, [expandedCardsInitialStates, getAllExpandedCardStepsFromStorage, navigateTo, state.expandedCardSteps, stepIdFromHash]);
  (0, _react.useEffect)(() => {
    /** Handle hash change and expand the target step.
     ** e.g.: https://localhost:5601/app/security/get_started#create_your_first_project
     **/
    if (stepIdFromHash) {
      const {
        matchedCard,
        matchedStep,
        matchedSection
      } = (0, _helpers.findCardSectionByStepId)(stepIdFromHash);
      const hasStepContent = matchedStep && matchedStep.description;
      if (hasStepContent && matchedCard && matchedStep && matchedSection) {
        var _state$expandedCardSt;
        // If the step is already expanded, do nothing
        if ((_state$expandedCardSt = state.expandedCardSteps[matchedCard.id]) !== null && _state$expandedCardSt !== void 0 && _state$expandedCardSt.expandedSteps.includes(matchedStep.id)) {
          return;
        }

        // The step is opened by navigation instead of clicking directly on the step. e.g.: clicking a stepLink
        // Toggle step and sync the expanded card step to storage & reducer
        onStepClicked({
          stepId: matchedStep.id,
          cardId: matchedCard.id,
          sectionId: matchedSection.id,
          isExpanded: true,
          trigger: 'navigation'
        });
        navigateTo({
          deepLinkId: _common.SecurityPageName.landing,
          path: `#${matchedStep.id}`
        });
      }
    }
  }, [navigateTo, onStepClicked, state.expandedCardSteps, stepIdFromHash]);
  return {
    state,
    onStepClicked,
    toggleTaskCompleteStatus,
    onProductSwitchChanged
  };
};
exports.useTogglePanel = useTogglePanel;