"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatNavigationTree = void 0;
var _fp = require("lodash/fp");
var _securitySolutionNavigation = require("@kbn/security-solution-navigation");
var _util = require("./util");
var _translations = require("../../common/translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatNavigationTree = (solutionNavLinks, bodyCategories, footerCategories) => {
  const [footerNavItems, bodyNavItems] = (0, _fp.partition)('isFooterLink', solutionNavLinks);
  const bodyChildren = addMainLinksPanelOpenerProp(formatNodesFromLinks(bodyNavItems, bodyCategories));
  return {
    body: [{
      type: 'navGroup',
      id: 'security_solution_nav',
      title: _translations.SOLUTION_NAME,
      icon: 'logoSecurity',
      breadcrumbStatus: 'hidden',
      defaultIsCollapsed: false,
      children: bodyChildren,
      isCollapsible: false
    }],
    footer: formatFooterNodesFromLinks(footerNavItems, footerCategories)
  };
};

// Body
exports.formatNavigationTree = formatNavigationTree;
const formatNodesFromLinks = (solutionNavLinks, parentCategories) => {
  const nodes = [];
  if (parentCategories !== null && parentCategories !== void 0 && parentCategories.length) {
    parentCategories.forEach(category => {
      nodes.push(...formatNodesFromLinksWithCategory(solutionNavLinks, category));
    }, []);
  } else {
    nodes.push(...formatNodesFromLinksWithoutCategory(solutionNavLinks));
  }
  return nodes;
};
const formatNodesFromLinksWithCategory = (solutionNavLinks, category) => {
  if (!(category !== null && category !== void 0 && category.linkIds)) {
    return [];
  }
  if (category.linkIds) {
    const children = category.linkIds.reduce((acc, linkId) => {
      const solutionNavLink = solutionNavLinks.find(({
        id
      }) => id === linkId);
      if (solutionNavLink != null) {
        acc.push(createNodeFromSolutionNavLink(solutionNavLink));
      }
      return acc;
    }, []);
    if (!children.length) {
      return [];
    }
    const id = (0, _securitySolutionNavigation.isTitleLinkCategory)(category) ? getCategoryIdFromLabel(category.label) : undefined;
    return [{
      id,
      ...((0, _securitySolutionNavigation.isTitleLinkCategory)(category) && {
        title: category.label
      }),
      breadcrumbStatus: 'hidden',
      children
    }];
  }
  return [];
};
const formatNodesFromLinksWithoutCategory = solutionNavLinks => solutionNavLinks.map(solutionNavLink => createNodeFromSolutionNavLink(solutionNavLink));
const createNodeFromSolutionNavLink = solutionNavLink => {
  const {
    id,
    title,
    links,
    categories,
    disabled
  } = solutionNavLink;
  const link = (0, _util.getNavLinkIdFromSolutionPageName)(id);
  const node = {
    id,
    link: link,
    title,
    ...((0, _util.isBreadcrumbHidden)(id) && {
      breadcrumbStatus: 'hidden'
    }),
    ...(disabled && {
      sideNavStatus: 'hidden'
    })
  };
  if (links !== null && links !== void 0 && links.length) {
    node.children = formatNodesFromLinks(links, categories);
  }
  return node;
};

// Footer

const formatFooterNodesFromLinks = (solutionNavLinks, parentCategories) => {
  const nodes = [];
  if (parentCategories !== null && parentCategories !== void 0 && parentCategories.length) {
    parentCategories.forEach(category => {
      if ((0, _securitySolutionNavigation.isSeparatorLinkCategory)(category)) {
        nodes.push(...category.linkIds.reduce((acc, linkId) => {
          const solutionNavLink = solutionNavLinks.find(({
            id
          }) => id === linkId);
          if (solutionNavLink != null) {
            acc.push({
              type: 'navItem',
              link: (0, _util.getNavLinkIdFromSolutionPageName)(solutionNavLink.id),
              title: solutionNavLink.title,
              icon: solutionNavLink.sideNavIcon
            });
          }
          return acc;
        }, []));
      }
      if ((0, _securitySolutionNavigation.isAccordionLinkCategory)(category)) {
        var _category$linkIds$red, _category$linkIds;
        nodes.push({
          type: 'navGroup',
          id: getCategoryIdFromLabel(category.label),
          title: category.label,
          icon: category.iconType,
          breadcrumbStatus: 'hidden',
          children: (_category$linkIds$red = (_category$linkIds = category.linkIds) === null || _category$linkIds === void 0 ? void 0 : _category$linkIds.reduce((acc, linkId) => {
            const solutionNavLink = solutionNavLinks.find(({
              id
            }) => id === linkId);
            if (solutionNavLink != null) {
              acc.push({
                title: solutionNavLink.title,
                link: (0, _util.getNavLinkIdFromSolutionPageName)(solutionNavLink.id)
              });
            }
            return acc;
          }, [])) !== null && _category$linkIds$red !== void 0 ? _category$linkIds$red : []
        });
      }
    }, []);
  }
  return nodes;
};

// Utils

const getCategoryIdFromLabel = label => `category-${label.toLowerCase().replace(' ', '_')}`;

/**
 * Adds the `renderAs: 'panelOpener'` prop to the main links that have children
 * This function expects all main links to be in nested groups to add the separation between them.
 * If these "separator" groups change this function will need to be updated.
 */
const addMainLinksPanelOpenerProp = nodes => nodes.map(node => {
  var _node$children;
  if ((_node$children = node.children) !== null && _node$children !== void 0 && _node$children.length) {
    return {
      ...node,
      children: node.children.map(child => ({
        ...child,
        ...(child.children && {
          renderAs: 'panelOpener'
        })
      }))
    };
  }
  return node;
});