"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSolutionNavLinks$ = void 0;
var _rxjs = require("rxjs");
var _securitySolutionNavigation = require("@kbn/security-solution-navigation");
var _links = require("@kbn/security-solution-navigation/links");
var _assets_links = require("./sections/assets_links");
var _ml_links = require("./sections/ml_links");
var _settings_links = require("./sections/settings_links");
var _dev_tools_links = require("./sections/dev_tools_links");
var _discover_links = require("./sections/discover_links");
var _util = require("../util");
var _investigations_links = require("./sections/investigations_links");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createSolutionNavLinks$ = (securityNavLinks$, core) => {
  const {
    chrome
  } = core;
  return (0, _rxjs.combineLatest)([securityNavLinks$, chrome.navLinks.getNavLinks$()]).pipe((0, _rxjs.debounceTime)(100),
  // avoid multiple calls in a short period of time
  (0, _rxjs.skipWhile)(([securityNavLinks, chromeNavLinks]) => securityNavLinks.length === 0 || chromeNavLinks.length === 0 // skip if not initialized
  ), (0, _rxjs.map)(([securityNavLinks]) => processNavLinks(securityNavLinks, chrome.navLinks)));
};

/**
 * Takes the security nav links and the chrome nav links and generates the project nav links
 * containing Security internal nav links and the external nav links (ML, Dev Tools, Project Settings, etc.)
 */
exports.createSolutionNavLinks$ = createSolutionNavLinks$;
const processNavLinks = (securityNavLinks, chromeNavLinks) => {
  const solutionNavLinks = [...securityNavLinks];

  // Discover. just pushing it
  solutionNavLinks.push(_discover_links.discoverNavLink);

  // Investigations. injecting external sub-links and categories definition to the landing
  const investigationsLinkIndex = solutionNavLinks.findIndex(({
    id
  }) => id === _securitySolutionNavigation.SecurityPageName.investigations);
  if (investigationsLinkIndex !== -1) {
    var _investigationNavLink;
    const investigationNavLink = solutionNavLinks[investigationsLinkIndex];
    solutionNavLinks[investigationsLinkIndex] = {
      ...investigationNavLink,
      links: [...((_investigationNavLink = investigationNavLink.links) !== null && _investigationNavLink !== void 0 ? _investigationNavLink : []), ..._investigations_links.investigationsNavLinks]
    };
  }

  // ML. injecting external sub-links and categories definition to the landing
  const mlLinkIndex = solutionNavLinks.findIndex(({
    id
  }) => id === _securitySolutionNavigation.SecurityPageName.mlLanding);
  if (mlLinkIndex !== -1) {
    solutionNavLinks[mlLinkIndex] = {
      ...solutionNavLinks[mlLinkIndex],
      categories: _ml_links.mlNavCategories,
      links: _ml_links.mlNavLinks
    };
  }

  // Assets, adding fleet external sub-links
  const assetsLinkIndex = solutionNavLinks.findIndex(({
    id
  }) => id === _securitySolutionNavigation.SecurityPageName.assets);
  if (assetsLinkIndex !== -1) {
    var _assetsNavLink$links;
    const assetsNavLink = solutionNavLinks[assetsLinkIndex];
    solutionNavLinks[assetsLinkIndex] = {
      ...assetsNavLink,
      links: [..._assets_links.assetsNavLinks, ...((_assetsNavLink$links = assetsNavLink.links) !== null && _assetsNavLink$links !== void 0 ? _assetsNavLink$links : [])] // adds fleet to the existing (endpoints and cloud) links
    };
  }

  // Dev Tools. just pushing it
  solutionNavLinks.push(_dev_tools_links.devToolsNavLink);
  solutionNavLinks.push(..._settings_links.settingsNavLinks);
  return filterDisabled(solutionNavLinks, chromeNavLinks);
};

/**
 * Filters out the disabled external kibana nav links from the project nav links.
 * Internal Security links are already filtered by the security_solution plugin appLinks.
 */
const filterDisabled = (solutionNavLinks, chromeNavLinks) => {
  return solutionNavLinks.reduce((filteredNavLinks, navLink) => {
    const {
      id,
      links
    } = navLink;
    if (!(0, _links.isSecurityId)(id)) {
      const navLinkId = (0, _util.getNavLinkIdFromSolutionPageName)(id);
      if (!chromeNavLinks.has(navLinkId)) {
        return filteredNavLinks;
      }
    }
    if (links) {
      filteredNavLinks.push({
        ...navLink,
        links: filterDisabled(links, chromeNavLinks)
      });
    } else {
      filteredNavLinks.push(navLink);
    }
    return filteredNavLinks;
  }, []);
};