"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EndpointActionGenerator = void 0;
var _lodash = require("lodash");
var _constants = require("../constants");
var _base_data_generator = require("./base_data_generator");
var _types = require("../types");
var _constants2 = require("../service/response_actions/constants");
var _get_file_download_id = require("../service/response_actions/get_file_download_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EndpointActionGenerator extends _base_data_generator.BaseDataGenerator {
  /** Generate a random endpoint Action request (isolate or unisolate) */
  generate(overrides = {}) {
    const timeStamp = overrides['@timestamp'] ? new Date(overrides['@timestamp']) : new Date();
    const doc = {
      '@timestamp': timeStamp.toISOString(),
      agent: {
        id: [this.seededUUIDv4()]
      },
      EndpointActions: {
        action_id: this.seededUUIDv4(),
        expiration: this.randomFutureDate(timeStamp),
        type: 'INPUT_ACTION',
        input_type: 'endpoint',
        data: {
          command: this.randomResponseActionCommand(),
          comment: this.randomString(15),
          parameters: undefined
        }
      },
      error: undefined,
      user: {
        id: this.randomUser()
      },
      rule: undefined
    };
    return (0, _lodash.merge)(doc, overrides);
  }
  generateActionEsHit(overrides = {}) {
    return Object.assign(this.toEsSearchHit(this.generate(overrides)), {
      _index: `.ds-${_constants.ENDPOINT_ACTIONS_DS}-some_namespace`
    });
  }

  /** Generates an endpoint action response */
  generateResponse(overrides = {}) {
    var _overrides$EndpointAc, _overrides$EndpointAc2, _overrides$EndpointAc3, _overrides$EndpointAc4, _overrides$EndpointAc5;
    const timeStamp = overrides['@timestamp'] ? new Date(overrides['@timestamp']) : new Date();
    const startedAtTimes = [];
    [2, 3, 5, 8, 13, 21].forEach(n => {
      startedAtTimes.push(timeStamp.setMinutes(-this.randomN(n)), timeStamp.setSeconds(-this.randomN(n)));
    });
    const command = (_overrides$EndpointAc = overrides === null || overrides === void 0 ? void 0 : (_overrides$EndpointAc2 = overrides.EndpointActions) === null || _overrides$EndpointAc2 === void 0 ? void 0 : (_overrides$EndpointAc3 = _overrides$EndpointAc2.data) === null || _overrides$EndpointAc3 === void 0 ? void 0 : _overrides$EndpointAc3.command) !== null && _overrides$EndpointAc !== void 0 ? _overrides$EndpointAc : this.randomResponseActionCommand();
    let output = overrides === null || overrides === void 0 ? void 0 : (_overrides$EndpointAc4 = overrides.EndpointActions) === null || _overrides$EndpointAc4 === void 0 ? void 0 : (_overrides$EndpointAc5 = _overrides$EndpointAc4.data) === null || _overrides$EndpointAc5 === void 0 ? void 0 : _overrides$EndpointAc5.output;
    if (command === 'get-file') {
      if (!output) {
        output = {
          type: 'json',
          content: {
            code: 'ra_get-file_success_done',
            zip_size: 123,
            contents: [{
              type: 'file',
              path: '/some/path/bad_file.txt',
              size: 1234,
              file_name: 'bad_file.txt',
              sha256: '9558c5cb39622e9b3653203e772b129d6c634e7dbd7af1b244352fc1d704601f'
            }]
          }
        };
      }
    }
    if (command === 'execute') {
      if (!output) {
        output = this.generateExecuteActionResponseOutput();
      }
    }
    if (command === 'upload' && !output) {
      let uploadOutput = output;
      if (overrides.error) {
        uploadOutput = {
          type: 'json',
          content: {
            code: 'ra_upload_some-error',
            path: '',
            disk_free_space: 0
          }
        };
      } else {
        uploadOutput = {
          type: 'json',
          content: {
            code: 'ra_upload_file-success',
            path: '/disk1/file/saved/here',
            disk_free_space: 4825566125475
          }
        };
      }
      output = uploadOutput;
    }
    return (0, _lodash.merge)({
      '@timestamp': timeStamp.toISOString(),
      agent: {
        id: this.seededUUIDv4()
      },
      EndpointActions: {
        action_id: this.seededUUIDv4(),
        completed_at: timeStamp.toISOString(),
        // randomly before a few hours/minutes/seconds later
        started_at: new Date(startedAtTimes[this.randomN(startedAtTimes.length)]).toISOString(),
        data: {
          command,
          comment: '',
          output
        }
      },
      error: undefined
    }, overrides);
  }
  generateResponseEsHit(overrides = {}) {
    return Object.assign(this.toEsSearchHit(this.generateResponse(overrides)), {
      _index: `.ds-${_constants.ENDPOINT_ACTION_RESPONSES_DS}-some_namespace-something`
    });
  }
  generateActionDetails(overrides = {}) {
    var _overrides$command;
    const details = {
      action: '123',
      agents: ['agent-a'],
      agentType: 'endpoint',
      command: 'isolate',
      completedAt: '2022-04-30T16:08:47.449Z',
      hosts: {
        'agent-a': {
          name: 'Host-agent-a'
        }
      },
      id: '123',
      isCompleted: true,
      isExpired: false,
      wasSuccessful: true,
      errors: undefined,
      startedAt: '2022-04-27T16:08:47.449Z',
      status: 'successful',
      comment: 'thisisacomment',
      createdBy: 'auserid',
      parameters: undefined,
      outputs: {},
      agentState: {
        'agent-a': {
          errors: undefined,
          isCompleted: true,
          completedAt: '2022-04-30T16:08:47.449Z',
          wasSuccessful: true
        }
      },
      alertIds: undefined,
      ruleId: undefined,
      ruleName: undefined
    };
    const command = (_overrides$command = overrides.command) !== null && _overrides$command !== void 0 ? _overrides$command : details.command;
    if (command === 'get-file') {
      if (!details.parameters) {
        details.parameters = {
          path: '/some/file.txt'
        };
      }
      if (!details.outputs || Object.keys(details.outputs).length === 0) {
        details.outputs = {
          [details.agents[0]]: {
            type: 'json',
            content: {
              code: 'ra_get-file_success',
              zip_size: 123,
              contents: [{
                path: '/some/file/txt',
                sha256: '1254',
                size: 1234,
                file_name: 'some-file.txt',
                type: 'file'
              }]
            }
          }
        };
      }
    }
    if (command === 'execute') {
      if (!details.parameters) {
        var _command, _overrides$parameters, _timeout, _overrides$parameters2;
        details.parameters = {
          command: (_command = (_overrides$parameters = overrides.parameters) === null || _overrides$parameters === void 0 ? void 0 : _overrides$parameters.command) !== null && _command !== void 0 ? _command : 'ls -al',
          timeout: (_timeout = (_overrides$parameters2 = overrides.parameters) === null || _overrides$parameters2 === void 0 ? void 0 : _overrides$parameters2.timeout) !== null && _timeout !== void 0 ? _timeout : _constants2.DEFAULT_EXECUTE_ACTION_TIMEOUT // 4hrs
        };
      }
      if (!details.outputs || Object.keys(details.outputs).length === 0) {
        var _overrides$outputs$de, _overrides$outputs, _overrides$outputs$de2;
        details.outputs = {
          [details.agents[0]]: this.generateExecuteActionResponseOutput({
            content: {
              output_file_id: (0, _get_file_download_id.getFileDownloadId)(details, details.agents[0]),
              ...((_overrides$outputs$de = (_overrides$outputs = overrides.outputs) === null || _overrides$outputs === void 0 ? void 0 : (_overrides$outputs$de2 = _overrides$outputs[details.agents[0]]) === null || _overrides$outputs$de2 === void 0 ? void 0 : _overrides$outputs$de2.content) !== null && _overrides$outputs$de !== void 0 ? _overrides$outputs$de : {})
            }
          })
        };
      }
    }
    if (command === 'upload') {
      const uploadActionDetails = details;
      uploadActionDetails.parameters = {
        file_id: 'file-x-y-z',
        file_name: 'foo.txt',
        file_size: 1234,
        file_sha256: 'file-hash-sha-256'
      };
      uploadActionDetails.outputs = {
        'agent-a': {
          type: 'json',
          content: {
            code: 'ra_upload_file-success',
            path: '/path/to/uploaded/file',
            disk_free_space: 1234567
          }
        }
      };
    }
    return (0, _lodash.merge)(details, overrides);
  }
  randomGetFileFailureCode() {
    return this.randomChoice(['ra_get-file_error_not-found', 'ra_get-file_error_is-directory', 'ra_get-file_error_invalid-input', 'ra_get-file_error_not-permitted', 'ra_get-file_error_too-big', 'ra_get-file_error_disk-quota', 'ra_get-file_error_processing', 'ra_get-file_error_upload-api-unreachable', 'ra_get-file_error_upload-timeout', 'ra_get-file_error_queue-timeout']);
  }
  generateActivityLogAction(overrides) {
    return (0, _lodash.merge)({
      type: _types.ActivityLogItemTypes.ACTION,
      item: {
        id: this.seededUUIDv4(),
        data: this.generate()
      }
    }, overrides);
  }
  generateActivityLogActionResponse(overrides) {
    var _overrides$item$data, _overrides$item;
    return (0, _lodash.merge)({
      type: _types.ActivityLogItemTypes.RESPONSE,
      item: {
        id: this.seededUUIDv4(),
        data: this.generateResponse({
          ...((_overrides$item$data = overrides === null || overrides === void 0 ? void 0 : (_overrides$item = overrides.item) === null || _overrides$item === void 0 ? void 0 : _overrides$item.data) !== null && _overrides$item$data !== void 0 ? _overrides$item$data : {})
        })
      }
    }, overrides);
  }
  generateAgentPendingActionsSummary(overrides = {}) {
    return (0, _lodash.merge)({
      agent_id: this.seededUUIDv4(),
      pending_actions: {
        isolate: 2,
        unisolate: 0
      }
    }, overrides);
  }
  generateExecuteActionResponseOutput(overrides) {
    return (0, _lodash.merge)({
      type: 'json',
      content: {
        code: 'ra_execute_success_done',
        stdout: this.randomChoice([this.randomString(1280), this.randomString(3580), `-rw-r--r--    1 elastic  staff      458 Jan 26 09:10 doc.txt\
          -rw-r--r--     1 elastic  staff  298 Feb  2 09:10 readme.md`]),
        stderr: this.randomChoice([this.randomString(1280), this.randomString(3580), `error line 1\
          error line 2\
          error line 3 that is quite very long and will be truncated, and should not be visible in the UI\
          errorline4thathasalotmoretextthatdoesnotendfortestingpurposesrepeatalotoftexthereandkeepaddingmoreandmoretextwithoutendtheideabeingthatwedonotuseperiodsorcommassothattheconsoleuiisunabletobreakthislinewithoutsomecssrulessowiththislineweshouldbeabletotestthatwithgenerateddata`]),
        stdout_truncated: true,
        stderr_truncated: true,
        shell_code: 0,
        shell: 'bash',
        cwd: this.randomChoice(['/some/path', '/a-very/long/path'.repeat(30)]),
        output_file_id: 'some-output-file-id',
        output_file_stdout_truncated: this.randomChoice([true, false]),
        output_file_stderr_truncated: this.randomChoice([true, false])
      }
    }, overrides);
  }
  randomFloat() {
    return this.random();
  }
  randomN(max) {
    return super.randomN(max);
  }
  randomResponseActionProcesses(n) {
    const numberOfEntries = n !== null && n !== void 0 ? n : this.randomChoice([20, 30, 40, 50]);
    const entries = [];
    for (let i = 0; i < numberOfEntries; i++) {
      entries.push({
        command: this.randomResponseActionProcessesCommand(),
        pid: this.randomN(1000).toString(),
        entity_id: this.randomString(50),
        user: this.randomUser()
      });
    }
    return entries;
  }
  randomResponseActionProcessesCommand() {
    const commands = ['/opt/cmd1', '/opt/cmd2', '/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3', '/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3'];
    return this.randomChoice(commands);
  }
  randomResponseActionCommand() {
    return this.randomChoice(_constants2.RESPONSE_ACTION_API_COMMANDS_NAMES);
  }
}
exports.EndpointActionGenerator = EndpointActionGenerator;