"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KQL_FILTER_MUTABLE_RULES = exports.KQL_FILTER_IMMUTABLE_RULES = exports.KQL_FILTER_ENABLED_RULES = exports.KQL_FILTER_DISABLED_RULES = void 0;
exports.convertRuleSearchTermToKQL = convertRuleSearchTermToKQL;
exports.convertRuleTagsToKQL = convertRuleTagsToKQL;
exports.convertRuleTypesToKQL = convertRuleTypesToKQL;
exports.convertRulesFilterToKQL = convertRulesFilterToKQL;
var _detection_engine = require("../../api/detection_engine");
var _kql = require("../../utils/kql");
var _rule_fields = require("./rule_fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const KQL_FILTER_IMMUTABLE_RULES = exports.KQL_FILTER_IMMUTABLE_RULES = `${_rule_fields.PARAMS_IMMUTABLE_FIELD}: true`;
const KQL_FILTER_MUTABLE_RULES = exports.KQL_FILTER_MUTABLE_RULES = `${_rule_fields.PARAMS_IMMUTABLE_FIELD}: false`;
const KQL_FILTER_ENABLED_RULES = exports.KQL_FILTER_ENABLED_RULES = `${_rule_fields.ENABLED_FIELD}: true`;
const KQL_FILTER_DISABLED_RULES = exports.KQL_FILTER_DISABLED_RULES = `${_rule_fields.ENABLED_FIELD}: false`;
/**
 * Convert rules filter options object to KQL query
 *
 * @param filterOptions desired filters (e.g. filter/sortField/sortOrder)
 *
 * @returns KQL string
 */
function convertRulesFilterToKQL({
  filter: searchTerm,
  showCustomRules,
  showElasticRules,
  enabled,
  tags,
  excludeRuleTypes = [],
  ruleExecutionStatus
}) {
  const kql = [];
  if (searchTerm !== null && searchTerm !== void 0 && searchTerm.length) {
    kql.push(`(${convertRuleSearchTermToKQL(searchTerm)})`);
  }
  if (showCustomRules && showElasticRules) {
    // if both showCustomRules && showElasticRules selected we omit filter, as it includes all existing rules
  } else if (showElasticRules) {
    kql.push(KQL_FILTER_IMMUTABLE_RULES);
  } else if (showCustomRules) {
    kql.push(KQL_FILTER_MUTABLE_RULES);
  }
  if (enabled !== undefined) {
    kql.push(enabled ? KQL_FILTER_ENABLED_RULES : KQL_FILTER_DISABLED_RULES);
  }
  if (tags !== null && tags !== void 0 && tags.length) {
    kql.push(convertRuleTagsToKQL(tags));
  }
  if (excludeRuleTypes.length) {
    kql.push(`NOT ${convertRuleTypesToKQL(excludeRuleTypes)}`);
  }
  if (ruleExecutionStatus === _detection_engine.RuleExecutionStatusEnum.succeeded) {
    kql.push(`${_rule_fields.LAST_RUN_OUTCOME_FIELD}: "succeeded"`);
  } else if (ruleExecutionStatus === _detection_engine.RuleExecutionStatusEnum['partial failure']) {
    kql.push(`${_rule_fields.LAST_RUN_OUTCOME_FIELD}: "warning"`);
  } else if (ruleExecutionStatus === _detection_engine.RuleExecutionStatusEnum.failed) {
    kql.push(`${_rule_fields.LAST_RUN_OUTCOME_FIELD}: "failed"`);
  }
  return kql.join(' AND ');
}
const SEARCHABLE_RULE_ATTRIBUTES = [_rule_fields.RULE_NAME_FIELD, _rule_fields.RULE_PARAMS_FIELDS.INDEX, _rule_fields.RULE_PARAMS_FIELDS.TACTIC_ID, _rule_fields.RULE_PARAMS_FIELDS.TACTIC_NAME, _rule_fields.RULE_PARAMS_FIELDS.TECHNIQUE_ID, _rule_fields.RULE_PARAMS_FIELDS.TECHNIQUE_NAME, _rule_fields.RULE_PARAMS_FIELDS.SUBTECHNIQUE_ID, _rule_fields.RULE_PARAMS_FIELDS.SUBTECHNIQUE_NAME];
function convertRuleSearchTermToKQL(searchTerm, attributes = SEARCHABLE_RULE_ATTRIBUTES) {
  return attributes.map(param => `${param}: ${(0, _kql.prepareKQLStringParam)(searchTerm)}`).join(' OR ');
}
function convertRuleTagsToKQL(tags) {
  return `${_rule_fields.TAGS_FIELD}:(${tags.map(_kql.prepareKQLStringParam).join(' AND ')})`;
}
function convertRuleTypesToKQL(ruleTypes) {
  return `${_rule_fields.PARAMS_TYPE_FIELD}: (${ruleTypes.map(_kql.prepareKQLStringParam).join(' OR ')})`;
}