"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRetriever = createRetriever;
exports.defineRoutes = defineRoutes;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/ml-response-stream/server");
var _events = require("./analytics/events");
var _fetch_query_source_fields = require("./lib/fetch_query_source_fields");
var _assist = require("./utils/assist");
var _conversational_chain = require("./lib/conversational_chain");
var _error_handler = require("./utils/error_handler");
var _types = require("./types");
var _get_chat_params = require("./lib/get_chat_params");
var _fetch_indices = require("./lib/fetch_indices");
var _is_not_nullish = require("../common/is_not_nullish");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createRetriever(esQuery) {
  return question => {
    try {
      const replacedQuery = esQuery.replace(/{query}/g, question.replace(/"/g, '\\"'));
      const query = JSON.parse(replacedQuery);
      return query;
    } catch (e) {
      throw Error(e);
    }
  };
}
function defineRoutes({
  logger,
  router,
  getStartServices
}) {
  router.post({
    path: _types.APIRoutes.POST_QUERY_SOURCE_FIELDS,
    validate: {
      body: _configSchema.schema.object({
        indices: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, (0, _error_handler.errorHandler)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      indices
    } = request.body;
    const fields = await (0, _fetch_query_source_fields.fetchFields)(client, indices);
    return response.ok({
      body: fields
    });
  }));
  router.post({
    path: _types.APIRoutes.POST_CHAT_MESSAGE,
    validate: {
      body: _configSchema.schema.object({
        data: _configSchema.schema.object({
          connector_id: _configSchema.schema.string(),
          indices: _configSchema.schema.string(),
          prompt: _configSchema.schema.string(),
          citations: _configSchema.schema.boolean(),
          elasticsearch_query: _configSchema.schema.string(),
          summarization_model: _configSchema.schema.maybe(_configSchema.schema.string()),
          doc_size: _configSchema.schema.number(),
          source_fields: _configSchema.schema.string()
        }),
        messages: _configSchema.schema.any()
      })
    }
  }, (0, _error_handler.errorHandler)(async (context, request, response) => {
    var _connector$config, _data$summarization_m;
    const [{
      analytics
    }, {
      actions
    }] = await getStartServices();
    const {
      client
    } = (await context.core).elasticsearch;
    const aiClient = (0, _assist.createAssist)({
      es_client: client.asCurrentUser
    });
    const {
      messages,
      data
    } = await request.body;
    const {
      chatModel,
      chatPrompt,
      connector
    } = await (0, _get_chat_params.getChatParams)({
      connectorId: data.connector_id,
      model: data.summarization_model,
      citations: data.citations,
      prompt: data.prompt
    }, {
      actions,
      logger,
      request
    });
    let sourceFields = {};
    try {
      sourceFields = JSON.parse(data.source_fields);
      sourceFields = Object.keys(sourceFields).reduce((acc, key) => {
        // @ts-ignore
        acc[key] = sourceFields[key][0];
        return acc;
      }, {});
    } catch (e) {
      logger.error('Failed to parse the source fields', e);
      throw Error(e);
    }
    const chain = (0, _conversational_chain.ConversationalChain)({
      model: chatModel,
      rag: {
        index: data.indices,
        retriever: createRetriever(data.elasticsearch_query),
        content_field: sourceFields,
        size: Number(data.doc_size)
      },
      prompt: chatPrompt
    });
    let stream;
    try {
      stream = await chain.stream(aiClient, messages);
    } catch (e) {
      logger.error('Failed to create the chat stream', e);
      if (typeof e === 'object') {
        return response.badRequest({
          body: {
            message: e.message
          }
        });
      }
      throw e;
    }
    const {
      end,
      push,
      responseWithHeaders
    } = (0, _server.streamFactory)(request.headers, logger);
    const reader = stream.getReader();
    const textDecoder = new TextDecoder();
    async function pushStreamUpdate() {
      reader.read().then(({
        done,
        value
      }) => {
        if (done) {
          end();
          return;
        }
        push(textDecoder.decode(value));
        pushStreamUpdate();
      });
    }
    pushStreamUpdate();
    analytics.reportEvent(_events.sendMessageEvent.eventType, {
      connectorType: connector.actionTypeId + ((_connector$config = connector.config) !== null && _connector$config !== void 0 && _connector$config.apiProvider ? `-${connector.config.apiProvider}` : ''),
      model: (_data$summarization_m = data.summarization_model) !== null && _data$summarization_m !== void 0 ? _data$summarization_m : '',
      isCitationsEnabled: data.citations
    });
    return response.ok(responseWithHeaders);
  }));
  router.post({
    path: _types.APIRoutes.POST_API_KEY,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        expiresInDays: _configSchema.schema.number(),
        indices: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, (0, _error_handler.errorHandler)(async (context, request, response) => {
    const {
      name,
      expiresInDays,
      indices
    } = request.body;
    const {
      client
    } = (await context.core).elasticsearch;
    const apiKey = await client.asCurrentUser.security.createApiKey({
      name,
      expiration: `${expiresInDays}d`,
      role_descriptors: {
        [`playground-${name}-role`]: {
          cluster: [],
          indices: [{
            names: indices,
            privileges: ['read']
          }]
        }
      }
    });
    return response.ok({
      body: {
        apiKey
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));

  // SECURITY: We don't apply any authorization tags to this route because all actions performed
  // on behalf of the user making the request and governed by the user's own cluster privileges.
  router.get({
    path: _types.APIRoutes.GET_INDICES,
    validate: {
      query: _configSchema.schema.object({
        search_query: _configSchema.schema.maybe(_configSchema.schema.string()),
        size: _configSchema.schema.number({
          defaultValue: 10,
          min: 0
        })
      })
    }
  }, (0, _error_handler.errorHandler)(async (context, request, response) => {
    const {
      search_query: searchQuery,
      size
    } = request.query;
    const {
      client: {
        asCurrentUser
      }
    } = (await context.core).elasticsearch;
    const {
      indexNames
    } = await (0, _fetch_indices.fetchIndices)(asCurrentUser, searchQuery);
    const indexNameSlice = indexNames.slice(0, size).filter(_is_not_nullish.isNotNullish);
    return response.ok({
      body: {
        indices: indexNameSlice
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
}