"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.openApiVersion = exports.generateOpenApiDocument = void 0;
var _coreHttpServer = require("@kbn/core-http-server");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _util = require("./util");
var _oas_converter = require("./oas_converter");
var _operation_id_counter = require("./operation_id_counter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const openApiVersion = exports.openApiVersion = '3.0.0';
const generateOpenApiDocument = (appRouters, opts) => {
  var _opts$tags;
  const converter = new _oas_converter.OasConverter();
  const getOpId = (0, _operation_id_counter.createOperationIdCounter)();
  const paths = {};
  for (const router of appRouters.routers) {
    const result = processRouter(router, converter, getOpId, opts.pathStartsWith);
    Object.assign(paths, result.paths);
  }
  for (const router of appRouters.versionedRouters) {
    const result = processVersionedRouter(router, converter, getOpId, opts.pathStartsWith);
    Object.assign(paths, result.paths);
  }
  return {
    openapi: openApiVersion,
    info: {
      title: opts.title,
      description: opts.description,
      version: opts.version
    },
    servers: [{
      url: opts.baseUrl
    }],
    paths,
    components: {
      ...converter.getSchemaComponents(),
      securitySchemes: {
        basicAuth: {
          type: 'http',
          scheme: 'basic'
        },
        apiKeyAuth: {
          type: 'apiKey',
          in: 'header',
          name: 'Authorization'
        }
      }
    },
    security: [{
      basicAuth: []
    }],
    tags: (_opts$tags = opts.tags) === null || _opts$tags === void 0 ? void 0 : _opts$tags.map(tag => ({
      name: tag
    })),
    externalDocs: opts.docsUrl ? {
      url: opts.docsUrl
    } : undefined
  };
};
exports.generateOpenApiDocument = generateOpenApiDocument;
const extractRequestBody = (route, converter) => {
  return route.handlers.reduce((acc, handler) => {
    const schemas = (0, _util.extractValidationSchemaFromVersionedHandler)(handler);
    if (!(schemas !== null && schemas !== void 0 && schemas.request)) return acc;
    const schema = converter.convert(schemas.request.body);
    return {
      ...acc,
      [(0, _util.getVersionedContentString)(handler.options.version)]: {
        schema
      }
    };
  }, {});
};
const extractVersionedResponses = (route, converter) => {
  return route.handlers.reduce((acc, handler) => {
    const schemas = (0, _util.extractValidationSchemaFromVersionedHandler)(handler);
    if (!(schemas !== null && schemas !== void 0 && schemas.response)) return acc;
    const statusCodes = Object.keys(schemas.response);
    for (const statusCode of statusCodes) {
      var _route$options$descri, _acc$statusCode;
      const maybeSchema = (0, _coreHttpRouterServerInternal.unwrapVersionedResponseBodyValidation)(schemas.response[statusCode].body);
      const schema = converter.convert(maybeSchema);
      acc[statusCode] = {
        ...acc[statusCode],
        description: (_route$options$descri = route.options.description) !== null && _route$options$descri !== void 0 ? _route$options$descri : 'No description',
        content: {
          ...((_acc$statusCode = acc[statusCode]) !== null && _acc$statusCode !== void 0 ? _acc$statusCode : {}).content,
          [(0, _util.getVersionedContentString)(handler.options.version)]: {
            schema
          }
        }
      };
    }
    return acc;
  }, {});
};
const prepareRoutes = (routes, pathStartsWith) => {
  return routes.filter(pathStartsWith ? route => route.path.startsWith(pathStartsWith) : () => true);
};
const processVersionedRouter = (appRouter, converter, getOpId, pathStartsWith) => {
  const routes = prepareRoutes(appRouter.getRoutes(), pathStartsWith);
  const paths = {};
  for (const route of routes) {
    const pathParams = (0, _util.getPathParameters)(route.path);
    /**
     * Note: for a given route we accept that route params and query params remain BWC
     *       so we only take the latest version of the params and query params, we also
     *       assume at this point that we are generating for serverless.
     */
    let parameters = [];
    const versions = route.handlers.map(({
      options: {
        version: v
      }
    }) => v).sort();
    const newestVersion = _coreHttpRouterServerInternal.versionHandlerResolvers.newest(versions);
    const handler = route.handlers.find(({
      options: {
        version: v
      }
    }) => v === newestVersion);
    const schemas = handler ? (0, _util.extractValidationSchemaFromVersionedHandler)(handler) : undefined;
    try {
      var _extractValidationSch, _extractValidationSch2;
      if (handler && schemas) {
        var _schemas$request, _schemas$request2;
        const reqParams = (_schemas$request = schemas.request) === null || _schemas$request === void 0 ? void 0 : _schemas$request.params;
        let pathObjects = [];
        let queryObjects = [];
        if (reqParams) {
          pathObjects = converter.convertPathParameters(reqParams, pathParams);
        }
        const reqQuery = (_schemas$request2 = schemas.request) === null || _schemas$request2 === void 0 ? void 0 : _schemas$request2.query;
        if (reqQuery) {
          queryObjects = converter.convertQuery(reqQuery);
        }
        parameters = [(0, _util.getVersionedHeaderParam)(newestVersion, versions), ...pathObjects, ...queryObjects];
      }
      const hasBody = Boolean(handler && ((_extractValidationSch = (0, _util.extractValidationSchemaFromVersionedHandler)(handler)) === null || _extractValidationSch === void 0 ? void 0 : (_extractValidationSch2 = _extractValidationSch.request) === null || _extractValidationSch2 === void 0 ? void 0 : _extractValidationSch2.body));
      const path = {
        [route.method]: {
          requestBody: hasBody ? {
            content: extractRequestBody(route, converter)
          } : undefined,
          responses: extractVersionedResponses(route, converter),
          parameters,
          operationId: getOpId(route.path)
        }
      };
      assignToPathsObject(paths, route.path, path);
    } catch (e) {
      // Enrich the error message with a bit more context
      e.message = `Error generating OpenAPI for route '${route.path}' using newest version '${newestVersion}': ${e.message}`;
      throw e;
    }
  }
  return {
    paths
  };
};
const extractResponses = (route, converter) => {
  const responses = {};
  if (!route.validationSchemas) return responses;
  const validationSchemas = (0, _coreHttpServer.getResponseValidation)(route.validationSchemas);
  return !!validationSchemas ? Object.entries(validationSchemas).reduce((acc, [statusCode, schema]) => {
    var _route$options$descri2, _acc$statusCode2;
    const oasSchema = converter.convert(schema.body());
    acc[statusCode] = {
      ...acc[statusCode],
      description: (_route$options$descri2 = route.options.description) !== null && _route$options$descri2 !== void 0 ? _route$options$descri2 : 'No description',
      content: {
        ...((_acc$statusCode2 = acc[statusCode]) !== null && _acc$statusCode2 !== void 0 ? _acc$statusCode2 : {}).content,
        [(0, _util.getVersionedContentString)(_coreHttpRouterServerInternal.ALLOWED_PUBLIC_VERSION)]: {
          schema: oasSchema
        }
      }
    };
    return acc;
  }, responses) : responses;
};
const processRouter = (appRouter, converter, getOpId, pathStartsWith) => {
  const routes = prepareRoutes(appRouter.getRoutes({
    excludeVersionedRoutes: true
  }), pathStartsWith);
  const paths = {};
  for (const route of routes) {
    try {
      const pathParams = (0, _util.getPathParameters)(route.path);
      const validationSchemas = (0, _util.extractValidationSchemaFromRoute)(route);
      let parameters = [];
      if (validationSchemas) {
        let pathObjects = [];
        let queryObjects = [];
        const reqParams = validationSchemas.params;
        if (reqParams) {
          pathObjects = converter.convertPathParameters(reqParams, pathParams);
        }
        const reqQuery = validationSchemas.query;
        if (reqQuery) {
          queryObjects = converter.convertQuery(reqQuery);
        }
        parameters = [(0, _util.getVersionedHeaderParam)(_coreHttpRouterServerInternal.ALLOWED_PUBLIC_VERSION, [_coreHttpRouterServerInternal.ALLOWED_PUBLIC_VERSION]), ...pathObjects, ...queryObjects];
      }
      const path = {
        [route.method]: {
          requestBody: !!(validationSchemas !== null && validationSchemas !== void 0 && validationSchemas.body) ? {
            content: {
              [(0, _util.getVersionedContentString)(_coreHttpRouterServerInternal.ALLOWED_PUBLIC_VERSION)]: {
                schema: converter.convert(validationSchemas.body)
              }
            }
          } : undefined,
          responses: extractResponses(route, converter),
          parameters,
          operationId: getOpId(route.path)
        }
      };
      assignToPathsObject(paths, route.path, path);
    } catch (e) {
      // Enrich the error message with a bit more context
      e.message = `Error generating OpenAPI for route '${route.path}': ${e.message}`;
      throw e;
    }
  }
  return {
    paths
  };
};
const assignToPathsObject = (paths, path, pathObject) => {
  const pathName = path.replace('?', '');
  paths[pathName] = {
    ...paths[pathName],
    ...pathObject
  };
};