"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ExecuteReportTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment"));
var _rxjs = _interopRequireWildcard(require("rxjs"));
var Rx = _rxjs;
var _promises = require("stream/promises");
var _promises2 = require("timers/promises");
var _reportingCommon = require("@kbn/reporting-common");
var _server = require("@kbn/task-manager-plugin/server");
var _ = require(".");
var _2 = require("..");
var _map_to_reporting_error = require("../../../common/errors/map_to_reporting_error");
var _store = require("../store");
var _error_logger = require("./error_logger");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isOutput(output) {
  return output.size != null;
}
async function finishedWithNoPendingCallbacks(stream) {
  await (0, _promises.finished)(stream, {
    readable: false
  });

  // Race condition workaround:
  // `finished(...)` will resolve while there's still pending callbacks in the writable part of the `stream`.
  // This introduces a race condition where the code continues before the writable part has completely finished.
  // The `pendingCallbacks` function is a hack to ensure that all pending callbacks have been called before continuing.
  // For more information, see: https://github.com/nodejs/node/issues/46170
  await async function pendingCallbacks(delay = 1) {
    if (stream._writableState.pendingcb > 0) {
      await (0, _promises2.setTimeout)(delay);
      await pendingCallbacks(delay < 32 ? delay * 2 : delay);
    }
  }();
}
function parseError(error) {
  if (error instanceof Error) {
    return {
      name: error.constructor.name,
      message: error.message,
      stack: error.stack,
      cause: error.cause
    };
  }
  return error;
}
class ExecuteReportTask {
  constructor(reporting, config, logger) {
    (0, _defineProperty2.default)(this, "TYPE", _.REPORTING_EXECUTE_TYPE);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "taskManagerStart", void 0);
    (0, _defineProperty2.default)(this, "kibanaId", void 0);
    (0, _defineProperty2.default)(this, "kibanaName", void 0);
    (0, _defineProperty2.default)(this, "exportTypesRegistry", void 0);
    (0, _defineProperty2.default)(this, "store", void 0);
    (0, _defineProperty2.default)(this, "eventTracker", void 0);
    this.reporting = reporting;
    this.config = config;
    this.logger = logger.get('runTask');
    this.exportTypesRegistry = this.reporting.getExportTypesRegistry();
  }

  /*
   * To be called from plugin start
   */
  async init(taskManager) {
    this.taskManagerStart = taskManager;
    const {
      reporting
    } = this;
    const {
      uuid,
      name
    } = reporting.getServerInfo();
    this.kibanaId = uuid;
    this.kibanaName = name;
  }

  /*
   * Async get the ReportingStore: it is only available after PluginStart
   */
  async getStore() {
    if (this.store) {
      return this.store;
    }
    const {
      store
    } = await this.reporting.getPluginStartDeps();
    this.store = store;
    return store;
  }
  getTaskManagerStart() {
    if (!this.taskManagerStart) {
      throw new Error('Reporting task runner has not been initialized!');
    }
    return this.taskManagerStart;
  }
  getEventTracker(report) {
    if (this.eventTracker) {
      return this.eventTracker;
    }
    const eventTracker = this.reporting.getEventTracker(report._id, report.jobtype, report.payload.objectType);
    this.eventTracker = eventTracker;
    return this.eventTracker;
  }
  getJobContentEncoding(jobType) {
    const exportType = this.exportTypesRegistry.getByJobType(jobType);
    return exportType.jobContentEncoding;
  }
  async _claimJob(task) {
    if (this.kibanaId == null) {
      throw new Error(`Kibana instance ID is undefined!`);
    }
    if (this.kibanaName == null) {
      throw new Error(`Kibana instance name is undefined!`);
    }
    const store = await this.getStore();
    const report = await store.findReportFromTask(task); // receives seq_no and primary_term

    if (report.status === 'completed') {
      throw new Error(`Can not claim the report job: it is already completed!`);
    }
    const m = (0, _moment.default)();

    // check if job has exceeded the configured maxAttempts
    const maxAttempts = this.getMaxAttempts();
    if (report.attempts >= maxAttempts) {
      let err;
      if (report.error && (0, _map_to_reporting_error.isExecutionError)(report.error)) {
        // We have an error stored from a previous attempts, so we'll use that
        // error to fail the job and return it to the user.
        const {
          error
        } = report;
        err = (0, _map_to_reporting_error.mapToReportingError)(error);
        err.stack = error.stack;
      } else {
        if (report.error && report.error instanceof Error) {
          (0, _error_logger.errorLogger)(this.logger, 'Error executing report', report.error);
        }
        err = new _reportingCommon.QueueTimeoutError(`Max attempts reached (${maxAttempts}). Queue timeout reached.`);
      }
      await this._failJob(report, err);
      throw err;
    }
    const queueTimeout = (0, _reportingCommon.durationToNumber)(this.config.queue.timeout);
    const startTime = m.toISOString();
    const expirationTime = m.add(queueTimeout).toISOString();
    const doc = {
      kibana_id: this.kibanaId,
      kibana_name: this.kibanaName,
      attempts: report.attempts + 1,
      max_attempts: maxAttempts,
      started_at: startTime,
      timeout: queueTimeout,
      process_expiration: expirationTime
    };
    const claimedReport = new _store.SavedReport({
      ...report,
      ...doc
    });
    this.logger.info(`Claiming ${claimedReport.jobtype} ${report._id} ` + `[_index: ${report._index}] ` + `[_seq_no: ${report._seq_no}] ` + `[_primary_term: ${report._primary_term}] ` + `[attempts: ${report.attempts}] ` + `[process_expiration: ${expirationTime}]`);

    // event tracking of claimed job
    const eventTracker = this.getEventTracker(report);
    const timeSinceCreation = Date.now() - new Date(report.created_at).valueOf();
    eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.claimJob({
      timeSinceCreation
    });
    const resp = await store.setReportClaimed(claimedReport, doc);
    claimedReport._seq_no = resp._seq_no;
    claimedReport._primary_term = resp._primary_term;
    return claimedReport;
  }
  async _failJob(report, error) {
    var _docOutput, _docOutput$error_code, _docOutput2, _error$message;
    const message = `Failing ${report.jobtype} job ${report._id}`;

    // log the error
    let docOutput;
    if (error) {
      (0, _error_logger.errorLogger)(this.logger, message, error);
      docOutput = this._formatOutput(error);
    } else {
      (0, _error_logger.errorLogger)(this.logger, message);
    }

    // update the report in the store
    const store = await this.getStore();
    const completedTime = (0, _moment.default)();
    const doc = {
      completed_at: completedTime.toISOString(),
      output: (_docOutput = docOutput) !== null && _docOutput !== void 0 ? _docOutput : null
    };

    // event tracking of failed job
    const eventTracker = this.getEventTracker(report);
    const timeSinceCreation = Date.now() - new Date(report.created_at).valueOf();
    eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.failJob({
      timeSinceCreation,
      errorCode: (_docOutput$error_code = (_docOutput2 = docOutput) === null || _docOutput2 === void 0 ? void 0 : _docOutput2.error_code) !== null && _docOutput$error_code !== void 0 ? _docOutput$error_code : 'unknown',
      errorMessage: (_error$message = error === null || error === void 0 ? void 0 : error.message) !== null && _error$message !== void 0 ? _error$message : 'unknown'
    });
    return await store.setReportFailed(report, doc);
  }
  async _saveExecutionError(report, failedToExecuteErr) {
    const message = `Saving execution error for ${report.jobtype} job ${report._id}`;
    const errorParsed = parseError(failedToExecuteErr);
    // log the error
    (0, _error_logger.errorLogger)(this.logger, message, failedToExecuteErr);

    // update the report in the store
    const store = await this.getStore();
    const doc = {
      output: null,
      error: errorParsed
    };
    return await store.setReportError(report, doc);
  }
  _formatOutput(output) {
    const docOutput = {};
    const unknownMime = null;
    if (isOutput(output)) {
      docOutput.content_type = output.content_type || unknownMime;
      docOutput.max_size_reached = output.max_size_reached;
      docOutput.csv_contains_formulas = output.csv_contains_formulas;
      docOutput.size = output.size;
      docOutput.warnings = output.warnings && output.warnings.length > 0 ? output.warnings : undefined;
      docOutput.error_code = output.error_code;
    } else {
      var _output$humanFriendly;
      const defaultOutput = null;
      docOutput.content = ((_output$humanFriendly = output.humanFriendlyMessage) === null || _output$humanFriendly === void 0 ? void 0 : _output$humanFriendly.call(output)) || output.toString() || defaultOutput;
      docOutput.content_type = unknownMime;
      docOutput.warnings = [output.toString()];
      docOutput.error_code = output.code;
      docOutput.size = typeof docOutput.content === 'string' ? docOutput.content.length : 0;
    }
    return docOutput;
  }
  async _performJob(task, taskInstanceFields, cancellationToken, stream) {
    const exportType = this.exportTypesRegistry.getByJobType(task.jobtype);
    if (!exportType) {
      throw new Error(`No export type from ${task.jobtype} found to execute report`);
    }
    // run the report
    // if workerFn doesn't finish before timeout, call the cancellationToken and throw an error
    const queueTimeout = (0, _reportingCommon.durationToNumber)(this.config.queue.timeout);
    return Rx.lastValueFrom(Rx.from(exportType.runTask(task.id, task.payload, taskInstanceFields, cancellationToken, stream)).pipe((0, _rxjs.timeout)(queueTimeout)) // throw an error if a value is not emitted before timeout
    );
  }
  async _completeJob(report, output) {
    var _output$metrics, _output$metrics2, _output$metrics3;
    let docId = `/${report._index}/_doc/${report._id}`;
    this.logger.debug(`Saving ${report.jobtype} to ${docId}.`);
    const completedTime = (0, _moment.default)();
    const docOutput = this._formatOutput(output);
    const store = await this.getStore();
    const doc = {
      completed_at: completedTime.toISOString(),
      metrics: output.metrics,
      output: docOutput
    };
    docId = `/${report._index}/_doc/${report._id}`;
    const resp = await store.setReportCompleted(report, doc);
    this.logger.info(`Saved ${report.jobtype} job ${docId}`);
    report._seq_no = resp._seq_no;
    report._primary_term = resp._primary_term;

    // event tracking of completed job
    const eventTracker = this.getEventTracker(report);
    const byteSize = docOutput.size;
    const timeSinceCreation = completedTime.valueOf() - new Date(report.created_at).valueOf();
    if (((_output$metrics = output.metrics) === null || _output$metrics === void 0 ? void 0 : _output$metrics.csv) != null) {
      var _output$metrics$csv$r;
      eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.completeJobCsv({
        byteSize,
        timeSinceCreation,
        csvRows: (_output$metrics$csv$r = output.metrics.csv.rows) !== null && _output$metrics$csv$r !== void 0 ? _output$metrics$csv$r : -1
      });
    } else if (((_output$metrics2 = output.metrics) === null || _output$metrics2 === void 0 ? void 0 : _output$metrics2.pdf) != null || ((_output$metrics3 = output.metrics) === null || _output$metrics3 === void 0 ? void 0 : _output$metrics3.png) != null) {
      var _report$payload$layou, _report$payload$layou2, _report$payload$layou3, _report$payload$layou4, _output$metrics$pdf$p, _output$metrics$pdf;
      const {
        width,
        height
      } = (_report$payload$layou = (_report$payload$layou2 = report.payload.layout) === null || _report$payload$layou2 === void 0 ? void 0 : _report$payload$layou2.dimensions) !== null && _report$payload$layou !== void 0 ? _report$payload$layou : {};
      eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.completeJobScreenshot({
        byteSize,
        timeSinceCreation,
        screenshotLayout: (_report$payload$layou3 = (_report$payload$layou4 = report.payload.layout) === null || _report$payload$layou4 === void 0 ? void 0 : _report$payload$layou4.id) !== null && _report$payload$layou3 !== void 0 ? _report$payload$layou3 : 'preserve_layout',
        numPages: (_output$metrics$pdf$p = (_output$metrics$pdf = output.metrics.pdf) === null || _output$metrics$pdf === void 0 ? void 0 : _output$metrics$pdf.pages) !== null && _output$metrics$pdf$p !== void 0 ? _output$metrics$pdf$p : -1,
        screenshotPixels: Math.round((width !== null && width !== void 0 ? width : 0) * (height !== null && height !== void 0 ? height : 0))
      });
    }
    return report;
  }

  // Generic is used to let TS infer the return type at call site.
  async throwIfKibanaShutsDown() {
    await Rx.firstValueFrom(this.reporting.getKibanaShutdown$());
    throw new _reportingCommon.KibanaShuttingDownError();
  }

  /*
   * Provides a TaskRunner for Task Manager
   */
  getTaskRunner() {
    // Keep a separate local stack for each task run
    return ({
      taskInstance
    }) => {
      let jobId;
      const cancellationToken = new _reportingCommon.CancellationToken();
      const {
        attempts: taskAttempts,
        params: reportTaskParams,
        retryAt: taskRetryAt,
        startedAt: taskStartedAt
      } = taskInstance;
      return {
        /*
         * Runs a reporting job
         * Claim job: Finds the report in ReportingStore, updates it to "processing"
         * Perform job: Gets the export type's runner, runs it with the job params
         * Complete job: Updates the report in ReportStore with the output from the runner
         * If any error happens, additional retry attempts may be picked up by a separate instance
         */
        run: async () => {
          let report;
          const isLastAttempt = taskAttempts >= this.getMaxAttempts();

          // find the job in the store and set status to processing
          const task = reportTaskParams;
          jobId = task === null || task === void 0 ? void 0 : task.id;
          try {
            if (!jobId) {
              throw new Error('Invalid report data provided in scheduled task!');
            }
            if (!isLastAttempt) {
              this.reporting.trackReport(jobId);
            }

            // Update job status to claimed
            report = await this._claimJob(task);
          } catch (failedToClaim) {
            // error claiming report - log the error
            // could be version conflict, or too many attempts or no longer connected to ES
            (0, _error_logger.errorLogger)(this.logger, `Error in claiming ${jobId}`, failedToClaim);
          }
          if (!report) {
            this.reporting.untrackReport(jobId);
            if (isLastAttempt) {
              (0, _error_logger.errorLogger)(this.logger, `Job ${jobId} failed too many times. Exiting...`);
              return;
            }
            const errorMessage = `Job ${jobId} could not be claimed. Exiting...`;
            (0, _error_logger.errorLogger)(this.logger, errorMessage);

            // Throw so Task manager can clean up the failed task
            throw new Error(errorMessage);
          }
          const {
            jobtype: jobType,
            attempts
          } = report;
          const maxAttempts = this.getMaxAttempts();
          this.logger.debug(`Starting ${jobType} report ${jobId}: attempt ${attempts} of ${maxAttempts}.`);
          this.logger.debug(`Reports running: ${this.reporting.countConcurrentReports()}.`);
          const eventLog = this.reporting.getEventLogger(new _store.Report({
            ...task,
            _id: task.id,
            _index: task.index
          }));
          try {
            var _output$metrics4;
            const jobContentEncoding = this.getJobContentEncoding(jobType);
            const stream = await (0, _2.getContentStream)(this.reporting, {
              id: report._id,
              index: report._index,
              if_primary_term: report._primary_term,
              if_seq_no: report._seq_no
            }, {
              encoding: jobContentEncoding === 'base64' ? 'base64' : 'raw'
            });
            eventLog.logExecutionStart();
            const output = await Promise.race([this._performJob(task, {
              retryAt: taskRetryAt,
              startedAt: taskStartedAt
            }, cancellationToken, stream), this.throwIfKibanaShutsDown()]);
            stream.end();
            await finishedWithNoPendingCallbacks(stream);
            report._seq_no = stream.getSeqNo();
            report._primary_term = stream.getPrimaryTerm();
            eventLog.logExecutionComplete({
              ...((_output$metrics4 = output.metrics) !== null && _output$metrics4 !== void 0 ? _output$metrics4 : {}),
              byteSize: stream.bytesWritten
            });
            if (output) {
              this.logger.debug(`Job output size: ${stream.bytesWritten} bytes.`);
              report = await this._completeJob(report, {
                ...output,
                size: stream.bytesWritten
              });
            }
            // untrack the report for concurrency awareness
            this.logger.debug(`Stopping ${jobId}.`);
          } catch (failedToExecuteErr) {
            eventLog.logError(failedToExecuteErr);
            await this._saveExecutionError(report, failedToExecuteErr).catch(failedToSaveError => {
              (0, _error_logger.errorLogger)(this.logger, `Error in saving execution error ${jobId}`, failedToSaveError);
            });
            cancellationToken.cancel();
            const error = (0, _map_to_reporting_error.mapToReportingError)(failedToExecuteErr);
            (0, _server.throwRetryableError)(error, new Date(Date.now() + _.TIME_BETWEEN_ATTEMPTS));
          } finally {
            this.reporting.untrackReport(jobId);
            this.logger.debug(`Reports running: ${this.reporting.countConcurrentReports()}.`);
          }
        },
        /*
         * Called by Task Manager to stop the report execution process in case
         * of timeout or server shutdown
         */
        cancel: async () => {
          if (jobId) {
            this.logger.warn(`Cancelling job ${jobId}...`);
          }
          cancellationToken.cancel();
        }
      };
    };
  }
  getMaxAttempts() {
    var _this$config$capture$;
    return (_this$config$capture$ = this.config.capture.maxAttempts) !== null && _this$config$capture$ !== void 0 ? _this$config$capture$ : 1;
  }
  getTaskDefinition() {
    // round up from ms to the nearest second
    const queueTimeout = Math.ceil((0, _reportingCommon.numberToDuration)(this.config.queue.timeout).asSeconds()) + 's';
    const maxConcurrency = this.config.queue.pollEnabled ? 1 : 0;
    const maxAttempts = this.getMaxAttempts();
    return {
      type: _.REPORTING_EXECUTE_TYPE,
      title: 'Reporting: execute job',
      createTaskRunner: this.getTaskRunner(),
      maxAttempts: maxAttempts + 1,
      // Add 1 so we get an extra attempt in case of failure during a Kibana restart
      timeout: queueTimeout,
      maxConcurrency
    };
  }
  async scheduleTask(params) {
    const taskInstance = {
      taskType: _.REPORTING_EXECUTE_TYPE,
      state: {},
      params
    };
    return await this.getTaskManagerStart().schedule(taskInstance);
  }
  getStatus() {
    if (this.taskManagerStart) {
      return _.ReportingTaskStatus.INITIALIZED;
    }
    return _.ReportingTaskStatus.UNINITIALIZED;
  }
}
exports.ExecuteReportTask = ExecuteReportTask;