"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createFlameGraph = createFlameGraph;
var _frame_group = require("./frame_group");
var _hash = require("./hash");
var _profiling = require("./profiling");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Base Flamegraph
 */

/** Elasticsearch flamegraph */

/**
 *
 * createFlameGraph combines the base flamegraph with CPU-intensive values.
 * This allows us to create a flamegraph in two steps (e.g. first on the server
 * and finally in the browser).
 * @param base BaseFlameGraph
 * @param showErrorFrames
 * @returns ElasticFlameGraph
 */
function createFlameGraph(base, showErrorFrames) {
  if (!showErrorFrames) {
    // This loop jumps over the error frames in the graph.
    // Error frames only appear as child nodes of the root frame.
    // Error frames only have a single child node.
    for (let i = 0; i < base.Edges[0].length; i++) {
      const childNodeID = base.Edges[0][i];
      if ((0, _profiling.isErrorFrame)(base.FrameType[childNodeID])) {
        base.Edges[0][i] = base.Edges[childNodeID][0];
      }
    }
  }
  const graph = {
    Size: base.Size,
    SamplingRate: base.SamplingRate,
    Edges: base.Edges,
    FileID: base.FileID,
    FrameType: base.FrameType,
    Inline: base.Inline,
    ExeFilename: base.ExeFilename,
    AddressOrLine: base.AddressOrLine,
    FunctionName: base.FunctionName,
    FunctionOffset: base.FunctionOffset,
    SourceFilename: base.SourceFilename,
    SourceLine: base.SourceLine,
    CountInclusive: base.CountInclusive,
    CountExclusive: base.CountExclusive,
    ID: new Array(base.Size),
    Label: new Array(base.Size),
    TotalSeconds: base.TotalSeconds,
    TotalSamples: base.TotalSamples,
    SelfCPU: base.SelfCPU,
    TotalCPU: base.TotalCPU,
    SelfAnnualCO2KgsItems: base.AnnualCO2TonsExclusive.map(_utils.convertTonsToKgs),
    TotalAnnualCO2KgsItems: base.AnnualCO2TonsInclusive.map(_utils.convertTonsToKgs),
    SelfAnnualCostsUSDItems: base.AnnualCostsUSDExclusive,
    TotalAnnualCostsUSDItems: base.AnnualCostsUSDInclusive
  };
  const rootFrameGroupID = (0, _frame_group.createFrameGroupID)(graph.FileID[0], graph.AddressOrLine[0], graph.ExeFilename[0], graph.SourceFilename[0], graph.FunctionName[0]);
  graph.ID[0] = (0, _hash.fnv1a64)(new TextEncoder().encode(rootFrameGroupID));
  const queue = [0];
  while (queue.length > 0) {
    const parent = queue.pop();
    for (const child of graph.Edges[parent]) {
      const frameGroupID = (0, _frame_group.createFrameGroupID)(graph.FileID[child], graph.AddressOrLine[child], graph.ExeFilename[child], graph.SourceFilename[child], graph.FunctionName[child]);
      const bytes = new TextEncoder().encode(graph.ID[parent] + frameGroupID);
      graph.ID[child] = (0, _hash.fnv1a64)(bytes);
      queue.push(child);
    }
  }
  graph.Label[0] = 'root: Represents 100% of CPU time.';
  for (let i = 1; i < graph.Size; i++) {
    const metadata = (0, _profiling.createStackFrameMetadata)({
      FileID: graph.FileID[i],
      FrameType: graph.FrameType[i],
      Inline: graph.Inline[i],
      ExeFileName: graph.ExeFilename[i],
      AddressOrLine: graph.AddressOrLine[i],
      FunctionName: graph.FunctionName[i],
      FunctionOffset: graph.FunctionOffset[i],
      SourceFilename: graph.SourceFilename[i],
      SourceLine: graph.SourceLine[i]
    });
    graph.Label[i] = (0, _profiling.getCalleeLabel)(metadata);
  }
  return graph;
}