"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerTopNFunctionsAPMTransactionsRoute = registerTopNFunctionsAPMTransactionsRoute;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/observability-plugin/server");
var _lodash = require("lodash");
var _ = require(".");
var _common = require("../../common");
var _handle_route_error_handler = require("../utils/handle_route_error_handler");
var _compat = require("./compat");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const querySchema = _configSchema.schema.object({
  timeFrom: _configSchema.schema.number(),
  timeTo: _configSchema.schema.number(),
  functionName: _configSchema.schema.string(),
  serviceNames: _configSchema.schema.arrayOf(_configSchema.schema.string())
});
function registerTopNFunctionsAPMTransactionsRoute({
  router,
  logger,
  dependencies: {
    start: {
      profilingDataAccess
    },
    setup: {
      apmDataAccess
    }
  }
}) {
  const paths = (0, _common.getRoutePaths)();
  router.get({
    path: paths.APMTransactions,
    options: {
      tags: ['access:profiling', 'access:apm'],
      timeout: {
        idleSocket: _.IDLE_SOCKET_TIMEOUT
      }
    },
    validate: {
      query: querySchema
    }
  }, async (context, request, response) => {
    try {
      if (!apmDataAccess) {
        return response.ok({
          body: []
        });
      }
      const core = await context.core;
      const {
        transaction: transactionIndices
      } = await apmDataAccess.getApmIndices(core.savedObjects.client);
      const esClient = await (0, _compat.getClient)(context);
      const {
        timeFrom,
        timeTo,
        functionName,
        serviceNames
      } = request.query;
      const startSecs = timeFrom / 1000;
      const endSecs = timeTo / 1000;
      const transactionsPerService = await Promise.all(serviceNames.slice(0, 5).map(async serviceName => {
        const apmFunctions = await profilingDataAccess.services.fetchESFunctions({
          core,
          esClient,
          query: {
            bool: {
              filter: [...(0, _server.termQuery)('service.name', serviceName), {
                range: {
                  ['@timestamp']: {
                    gte: String(startSecs),
                    lt: String(endSecs),
                    format: 'epoch_second'
                  }
                }
              }]
            }
          },
          aggregationField: 'transaction.name',
          indices: transactionIndices.split(','),
          stacktraceIdsField: 'transaction.profiler_stack_trace_ids',
          limit: 1000,
          totalSeconds: endSecs - startSecs
        });
        const apmFunction = apmFunctions.TopN.find(topNFunction => topNFunction.Frame.FunctionName === functionName);
        if (apmFunction !== null && apmFunction !== void 0 && apmFunction.subGroups) {
          const subGroups = apmFunction.subGroups;
          return {
            serviceName,
            transactions: Object.keys(subGroups).map(key => ({
              name: key,
              samples: subGroups[key]
            }))
          };
        }
      }));
      const transactionsGroupedByService = (0, _lodash.keyBy)(transactionsPerService, 'serviceName');
      return response.ok({
        body: transactionsGroupedByService
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while fetching TopN functions'
      });
    }
  });
}