"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createCustomThresholdExecutor = void 0;
var _lodash = require("lodash");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _common = require("@kbn/alerting-plugin/common");
var _server = require("@kbn/alerting-plugin/server");
var _get_values = require("./lib/get_values");
var _common2 = require("../../../../common");
var _get_view_in_app_url = require("../../../../common/custom_threshold_rule/get_view_in_app_url");
var _constants = require("./constants");
var _types = require("./types");
var _messages = require("./messages");
var _utils = require("./utils");
var _format_alert_result = require("./lib/format_alert_result");
var _evaluate_rule = require("./lib/evaluate_rule");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createCustomThresholdExecutor = ({
  basePath,
  logger,
  config,
  locators: {
    logsExplorerLocator
  }
}) => async function (options) {
  var _state$searchConfigur, _alertsClient$getReco;
  const startTime = Date.now();
  const {
    services,
    params,
    state,
    startedAt,
    executionId,
    spaceId,
    rule: {
      id: ruleId
    },
    getTimeRange
  } = options;
  const {
    criteria
  } = params;
  if (criteria.length === 0) throw new Error('Cannot execute an alert with 0 conditions');
  const thresholdLogger = (0, _utils.createScopedLogger)(logger, 'thresholdRule', {
    alertId: ruleId,
    executionId
  });
  const {
    searchSourceClient,
    alertsClient
  } = services;
  if (!alertsClient) {
    throw new _server.AlertsClientError();
  }
  const {
    alertOnNoData,
    alertOnGroupDisappear: _alertOnGroupDisappear
  } = params;

  // For backwards-compatibility, interpret undefined alertOnGroupDisappear as true
  const alertOnGroupDisappear = _alertOnGroupDisappear !== false;
  const compositeSize = config.customThresholdRule.groupByPageSize;
  const queryIsSame = (0, _lodash.isEqual)((_state$searchConfigur = state.searchConfiguration) === null || _state$searchConfigur === void 0 ? void 0 : _state$searchConfigur.query.query, params.searchConfiguration.query.query);
  const groupByIsSame = (0, _lodash.isEqual)(state.groupBy, params.groupBy);
  const previousMissingGroups = alertOnGroupDisappear && queryIsSame && groupByIsSame && state.missingGroups ? state.missingGroups.filter(missingGroup =>
  // We use isTrackedAlert to remove missing groups that are untracked by the user
  alertsClient.isTrackedAlert(missingGroup.key)) : [];
  const initialSearchSource = await searchSourceClient.create(params.searchConfiguration);
  const dataView = initialSearchSource.getField('index');
  const {
    id: dataViewId,
    timeFieldName
  } = dataView;
  const dataViewIndexPattern = dataView.getIndexPattern();
  const dataViewName = dataView.getName();
  if (!dataViewIndexPattern) {
    throw new Error('No matched data view');
  } else if (!timeFieldName) {
    throw new Error('The selected data view does not have a timestamp field');
  }

  // Calculate initial start and end date with no time window, as each criterion has its own time window
  const {
    dateStart,
    dateEnd
  } = getTimeRange();
  const alertResults = await (0, _evaluate_rule.evaluateRule)(services.scopedClusterClient.asCurrentUser, params, dataViewIndexPattern, timeFieldName, compositeSize, alertOnGroupDisappear, logger, {
    end: dateEnd,
    start: dateStart
  }, state.lastRunTimestamp, previousMissingGroups);
  const resultGroupSet = new Set();
  for (const resultSet of alertResults) {
    for (const group of Object.keys(resultSet)) {
      resultGroupSet.add(group);
    }
  }
  const groupByKeysObjectMapping = (0, _utils.getFormattedGroupBy)(params.groupBy, resultGroupSet);
  const groupArray = [...resultGroupSet];
  const nextMissingGroups = new Set();
  const hasGroups = !(0, _lodash.isEqual)(groupArray, [_constants.UNGROUPED_FACTORY_KEY]);
  let scheduledActionsCount = 0;
  const alertLimit = alertsClient.getAlertLimitValue();
  let hasReachedLimit = false;

  // The key of `groupArray` is the alert instance ID.
  for (const group of groupArray) {
    if (scheduledActionsCount >= alertLimit) {
      // need to set this so that warning is displayed in the UI and in the logs
      hasReachedLimit = true;
      break; // once limit is reached, we break out of the loop and don't schedule any more alerts
    }

    // AND logic; all criteria must be across the threshold
    const shouldAlertFire = alertResults.every(result => {
      var _result$group;
      return (_result$group = result[group]) === null || _result$group === void 0 ? void 0 : _result$group.shouldFire;
    });
    // AND logic; because we need to evaluate all criteria, if one of them reports no data then the
    // whole alert is in a No Data/Error state
    const isNoData = alertResults.some(result => {
      var _result$group2;
      return (_result$group2 = result[group]) === null || _result$group2 === void 0 ? void 0 : _result$group2.isNoData;
    });
    if (isNoData && group !== _constants.UNGROUPED_FACTORY_KEY) {
      nextMissingGroups.add({
        key: group,
        bucketKey: alertResults[0][group].bucketKey
      });
    }
    const nextState = isNoData ? _types.AlertStates.NO_DATA : shouldAlertFire ? _types.AlertStates.ALERT : _types.AlertStates.OK;
    let reason;
    if (nextState === _types.AlertStates.ALERT) {
      reason = (0, _messages.buildFiredAlertReason)(alertResults, group, dataViewName);
    }

    /* NO DATA STATE HANDLING
     *
     * - `alertOnNoData` does not indicate IF the alert's next state is No Data, but whether or not the user WANTS TO BE ALERTED
     *   if the state were No Data.
     * - `alertOnGroupDisappear`, on the other hand, determines whether or not it's possible to return a No Data state
     *   when a group disappears.
     *
     * This means we need to handle the possibility that `alertOnNoData` is false, but `alertOnGroupDisappear` is true
     *
     * nextState === NO_DATA would be true on both { '*': No Data } or, e.g. { 'a': No Data, 'b': OK, 'c': OK }, but if the user
     * has for some reason disabled `alertOnNoData` and left `alertOnGroupDisappear` enabled, they would only care about the latter
     * possibility. In this case, use hasGroups to determine whether to alert on a potential No Data state
     *
     * If `alertOnNoData` is true but `alertOnGroupDisappear` is false, we don't need to worry about the {a, b, c} possibility.
     * At this point in the function, a false `alertOnGroupDisappear` would already have prevented group 'a' from being evaluated at all.
     */
    if (alertOnNoData || alertOnGroupDisappear && hasGroups) {
      // In the previous line we've determined if the user is interested in No Data states, so only now do we actually
      // check to see if a No Data state has occurred
      if (nextState === _types.AlertStates.NO_DATA) {
        reason = alertResults.filter(result => {
          var _result$group3;
          return (_result$group3 = result[group]) === null || _result$group3 === void 0 ? void 0 : _result$group3.isNoData;
        }).map(result => (0, _messages.buildNoDataAlertReason)({
          ...result[group],
          label: (0, _format_alert_result.getLabel)(result[group]),
          group
        })).join('\n');
      }
    }
    if (reason) {
      var _alertResults$0$group, _additionalContext$ta, _params$searchConfigu, _params$searchConfigu2, _params$searchConfigu3;
      const timestamp = startedAt.toISOString();
      const threshold = (0, _get_values.getThreshold)(criteria);
      const evaluationValues = (0, _get_values.getEvaluationValues)(alertResults, group);
      const actionGroupId = nextState === _types.AlertStates.OK ? _common.RecoveredActionGroup.id : nextState === _types.AlertStates.NO_DATA ? _constants.NO_DATA_ACTIONS_ID : _constants.FIRED_ACTIONS_ID;
      const additionalContext = (0, _utils.hasAdditionalContext)(params.groupBy, _utils.validGroupByForContext) ? alertResults && alertResults.length > 0 ? (_alertResults$0$group = alertResults[0][group].context) !== null && _alertResults$0$group !== void 0 ? _alertResults$0$group : {} : {} : {};
      additionalContext.tags = Array.from(new Set([...((_additionalContext$ta = additionalContext.tags) !== null && _additionalContext$ta !== void 0 ? _additionalContext$ta : []), ...options.rule.tags]));
      const groups = groupByKeysObjectMapping[group];
      const {
        uuid,
        start
      } = alertsClient.report({
        id: `${group}`,
        actionGroup: actionGroupId,
        payload: {
          [_ruleDataUtils.ALERT_REASON]: reason,
          [_ruleDataUtils.ALERT_EVALUATION_VALUES]: evaluationValues,
          [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: threshold,
          [_ruleDataUtils.ALERT_GROUP]: groups,
          ...(0, _utils.flattenAdditionalContext)(additionalContext)
        }
      });
      const indexedStartedAt = start !== null && start !== void 0 ? start : startedAt.toISOString();
      scheduledActionsCount++;
      alertsClient.setAlertData({
        id: `${group}`,
        context: {
          alertDetailsUrl: (0, _common2.getAlertDetailsUrl)(basePath, spaceId, uuid),
          group: groupByKeysObjectMapping[group],
          reason,
          timestamp,
          value: alertResults.map(result => {
            const evaluation = result[group];
            if (!evaluation) {
              return null;
            }
            return (0, _format_alert_result.formatAlertResult)(evaluation).currentValue;
          }),
          viewInAppUrl: (0, _get_view_in_app_url.getViewInAppUrl)({
            dataViewId: (_params$searchConfigu = (_params$searchConfigu2 = params.searchConfiguration) === null || _params$searchConfigu2 === void 0 ? void 0 : (_params$searchConfigu3 = _params$searchConfigu2.index) === null || _params$searchConfigu3 === void 0 ? void 0 : _params$searchConfigu3.title) !== null && _params$searchConfigu !== void 0 ? _params$searchConfigu : dataViewId,
            groups,
            logsExplorerLocator,
            metrics: alertResults.length === 1 ? alertResults[0][group].metrics : [],
            searchConfiguration: params.searchConfiguration,
            startedAt: indexedStartedAt
          }),
          ...additionalContext
        }
      });
    }
  }
  alertsClient.setAlertLimitReached(hasReachedLimit);
  const recoveredAlerts = (_alertsClient$getReco = alertsClient.getRecoveredAlerts()) !== null && _alertsClient$getReco !== void 0 ? _alertsClient$getReco : [];
  const groupByKeysObjectForRecovered = (0, _utils.getFormattedGroupBy)(params.groupBy, new Set(recoveredAlerts.map(recoveredAlert => recoveredAlert.alert.getId())));
  for (const recoveredAlert of recoveredAlerts) {
    var _recoveredAlert$alert, _params$criteria$;
    const recoveredAlertId = recoveredAlert.alert.getId();
    const alertUuid = recoveredAlert.alert.getUuid();
    const indexedStartedAt = (_recoveredAlert$alert = recoveredAlert.alert.getStart()) !== null && _recoveredAlert$alert !== void 0 ? _recoveredAlert$alert : startedAt.toISOString();
    const group = groupByKeysObjectForRecovered[recoveredAlertId];
    const alertHits = recoveredAlert.hit;
    const additionalContext = (0, _utils.getContextForRecoveredAlerts)(alertHits);
    const context = {
      alertDetailsUrl: (0, _common2.getAlertDetailsUrl)(basePath, spaceId, alertUuid),
      group,
      timestamp: startedAt.toISOString(),
      viewInAppUrl: (0, _get_view_in_app_url.getViewInAppUrl)({
        dataViewId,
        groups: group,
        logsExplorerLocator,
        metrics: (_params$criteria$ = params.criteria[0]) === null || _params$criteria$ === void 0 ? void 0 : _params$criteria$.metrics,
        searchConfiguration: params.searchConfiguration,
        startedAt: indexedStartedAt
      }),
      ...additionalContext
    };
    alertsClient.setAlertData({
      id: recoveredAlertId,
      context
    });
  }
  const stopTime = Date.now();
  thresholdLogger.debug(`Scheduled ${scheduledActionsCount} actions in ${stopTime - startTime}ms`);
  return {
    state: {
      lastRunTimestamp: startedAt.valueOf(),
      missingGroups: [...nextMissingGroups],
      groupBy: params.groupBy,
      searchConfiguration: params.searchConfiguration
    }
  };
};
exports.createCustomThresholdExecutor = createCustomThresholdExecutor;