"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mapRuleParamsWithFlyout = void 0;
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _i18n = require("@kbn/i18n");
var _common = require("../../../../common");
var _formatters = require("../../../../common/custom_threshold_rule/formatters");
var _metric_value_formatter = require("../../../../common/custom_threshold_rule/metric_value_formatter");
var _snapshot_metric_formats = require("../../../../common/custom_threshold_rule/formatters/snapshot_metric_formats");
var _alert_details = require("../../../pages/alert_details/alert_details");
var _is_fields_same_type = require("./is_fields_same_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getPctAboveThreshold = (observedValue, threshold) => {
  if (!observedValue || !threshold || threshold.length > 1 || threshold[0] <= 0) return '';
  return _i18n.i18n.translate('xpack.observability.alertFlyout.overview.aboveThresholdLabel', {
    defaultMessage: ' ({pctValue}% above the threshold)',
    values: {
      pctValue: parseFloat(((observedValue - threshold[0]) * 100 / threshold[0]).toFixed(2))
    }
  });
};
const mapRuleParamsWithFlyout = alert => {
  const ruleParams = alert.fields[_ruleDataUtils.ALERT_RULE_PARAMETERS];
  if (!ruleParams) return;
  const ruleCriteria = ruleParams === null || ruleParams === void 0 ? void 0 : ruleParams.criteria;
  const ruleId = alert.fields[_ruleDataUtils.ALERT_RULE_TYPE_ID];
  const observedValues = alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUES] || [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]];
  switch (ruleId) {
    case _ruleDataUtils.OBSERVABILITY_THRESHOLD_RULE_TYPE_ID:
      return observedValues.map((observedValue, metricIndex) => {
        const criteria = ruleCriteria[metricIndex];
        const fields = criteria.metrics.map(metric => metric.field || 'COUNT_AGG');
        const comparator = criteria.comparator;
        const threshold = criteria.threshold;
        const isSameFieldsType = (0, _is_fields_same_type.isFieldsSameType)(fields);
        const formattedValue = (0, _metric_value_formatter.metricValueFormatter)(observedValue, isSameFieldsType ? fields[0] : 'noType');
        const thresholdFormattedAsString = threshold.map(thresholdWithRange => (0, _metric_value_formatter.metricValueFormatter)(thresholdWithRange, isSameFieldsType ? fields[0] : 'noType')).join(' AND ');
        return {
          observedValue: formattedValue,
          threshold: thresholdFormattedAsString,
          comparator,
          pctAboveThreshold: getPctAboveThreshold(observedValue, threshold)
        };
      });
    case _alert_details.METRIC_THRESHOLD_ALERT_TYPE_ID:
      return observedValues.map((observedValue, metricIndex) => {
        const criteria = ruleCriteria[metricIndex];
        let fields = [];
        const metric = criteria.metric;
        const customMetric = criteria.customMetrics;
        if (metric) fields = [metric];
        if (customMetric && customMetric.length) fields = customMetric.map(cMetric => cMetric.field);
        const comparator = criteria.comparator;
        const threshold = criteria.threshold;
        const isSameFieldsType = (0, _is_fields_same_type.isFieldsSameType)(fields);
        const formattedValue = (0, _metric_value_formatter.metricValueFormatter)(observedValue, isSameFieldsType ? fields[0] : 'noType');
        const thresholdFormattedAsString = threshold.map(thresholdWithRange => (0, _metric_value_formatter.metricValueFormatter)(thresholdWithRange, isSameFieldsType ? fields[0] : 'noType')).join(' AND ');
        return {
          observedValue: formattedValue,
          threshold: thresholdFormattedAsString,
          comparator,
          pctAboveThreshold: getPctAboveThreshold(observedValue, threshold)
        };
      });
    case _ruleDataUtils.METRIC_INVENTORY_THRESHOLD_ALERT_TYPE_ID:
      return observedValues.map((observedValue, metricIndex) => {
        const criteria = ruleCriteria[metricIndex];
        const infraType = _snapshot_metric_formats.METRIC_FORMATTERS[criteria.metric].formatter;
        const formatter = (0, _formatters.createFormatter)(infraType);
        const comparator = criteria.comparator;
        const threshold = criteria.threshold;
        const formatThreshold = threshold.map(v => {
          if (infraType === 'percent') {
            v = Number(v) / 100;
          }
          if (infraType === 'bits') {
            v = Number(v) / 8;
          }
          return v;
        });
        return {
          observedValue: formatter(observedValue),
          threshold: formatThreshold.map(formatter),
          comparator,
          pctAboveThreshold: getPctAboveThreshold(observedValue, formatThreshold)
        };
      });
    case _ruleDataUtils.LOG_THRESHOLD_ALERT_TYPE_ID:
      const {
        comparator
      } = ruleParams === null || ruleParams === void 0 ? void 0 : ruleParams.count;
      const flyoutMap = {
        observedValue: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]],
        threshold: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]],
        comparator,
        pctAboveThreshold: getPctAboveThreshold(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE], [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]])
      };
      return [flyoutMap];
    case _ruleDataUtils.ApmRuleType.ErrorCount:
      const APMFlyoutMapErrorCount = {
        observedValue: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]],
        threshold: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]],
        comparator: '>',
        pctAboveThreshold: getPctAboveThreshold(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE], [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]])
      };
      return [APMFlyoutMapErrorCount];
    case _ruleDataUtils.ApmRuleType.TransactionErrorRate:
      const APMFlyoutMapTransactionErrorRate = {
        observedValue: [(0, _common.asPercent)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE], 100)],
        threshold: [(0, _common.asPercent)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD], 100)],
        comparator: '>',
        pctAboveThreshold: getPctAboveThreshold(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE], [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]])
      };
      return [APMFlyoutMapTransactionErrorRate];
    case _ruleDataUtils.ApmRuleType.TransactionDuration:
      const APMFlyoutMapTransactionDuration = {
        observedValue: [(0, _common.asDuration)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE])],
        threshold: [(0, _common.asDuration)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD])],
        comparator: '>',
        pctAboveThreshold: getPctAboveThreshold(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE], [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]])
      };
      return [APMFlyoutMapTransactionDuration];
    case '.es-query':
      const {
        thresholdComparator
      } = ruleParams;
      const ESQueryFlyoutMap = {
        observedValue: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]],
        threshold: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]],
        comparator: thresholdComparator,
        pctAboveThreshold: getPctAboveThreshold(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE], [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]])
      };
      return [ESQueryFlyoutMap];
    case _ruleDataUtils.SLO_BURN_RATE_RULE_TYPE_ID:
      const SLOBurnRateFlyoutMap = {
        observedValue: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]],
        threshold: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]],
        comparator: '>',
        pctAboveThreshold: getPctAboveThreshold(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE], [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]])
      };
      return [SLOBurnRateFlyoutMap];
    default:
      return [];
  }
};
exports.mapRuleParamsWithFlyout = mapRuleParamsWithFlyout;