"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.flushBuffer = flushBuffer;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _conversation_complete = require("../../../common/conversation_complete");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// The Cloud proxy currently buffers 4kb or 8kb of data until flushing.
// This decreases the responsiveness of the streamed response,
// so we manually insert some data every 250ms if needed to force it
// to flush.

function flushBuffer(isCloud) {
  if (!isCloud) {
    return _rxjs.identity;
  }
  return source => new _rxjs.Observable(subscriber => {
    const cloudProxyBufferSize = 4096;
    let currentBufferSize = 0;
    const flushBufferIfNeeded = () => {
      if (currentBufferSize && currentBufferSize <= cloudProxyBufferSize) {
        subscriber.next({
          data: (0, _lodash.repeat)('0', cloudProxyBufferSize * 2),
          type: _conversation_complete.StreamingChatResponseEventType.BufferFlush
        });
        currentBufferSize = 0;
      }
    };
    const intervalId = setInterval(flushBufferIfNeeded, 250);
    source.subscribe({
      next: value => {
        currentBufferSize = currentBufferSize <= cloudProxyBufferSize ? JSON.stringify(value).length + currentBufferSize : cloudProxyBufferSize;
        subscriber.next(value);
      },
      error: error => {
        clearInterval(intervalId);
        subscriber.error(error);
      },
      complete: () => {
        clearInterval(intervalId);
        subscriber.complete();
      }
    });
  });
}