"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.eventstreamSerdeIntoObservable = eventstreamSerdeIntoObservable;
var _eventstreamSerdeNode = require("@smithy/eventstream-serde-node");
var _utilUtf = require("@smithy/util-utf8");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _util = require("util");
var _conversation_complete = require("../../../common/conversation_complete");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// AWS uses SerDe to send over serialized data, so we use their
// @smithy library to parse the stream data

function eventstreamSerdeIntoObservable(readable, logger) {
  return new _rxjs.Observable(subscriber => {
    const marshaller = new _eventstreamSerdeNode.EventStreamMarshaller({
      utf8Encoder: _utilUtf.toUtf8,
      utf8Decoder: _utilUtf.fromUtf8
    });
    async function processStream() {
      for await (const chunk of marshaller.deserialize(readable, _lodash.identity)) {
        if (chunk) {
          subscriber.next(chunk);
        }
      }
    }
    processStream().then(() => {
      subscriber.complete();
    }, error => {
      if (!(error instanceof Error)) {
        try {
          const exceptionType = error.headers[':exception-type'].value;
          const body = (0, _utilUtf.toUtf8)(error.body);
          let message = 'Encountered error in Bedrock stream of type ' + exceptionType;
          try {
            message += '\n' + JSON.parse(body).message;
          } catch (parseError) {
            logger.error(`Could not parse message from stream error`);
            logger.error((0, _util.inspect)(body));
          }
          error = (0, _conversation_complete.createInternalServerError)(message);
        } catch (decodeError) {
          logger.error('Encountered unparsable error in Bedrock stream');
          logger.error((0, _util.inspect)(decodeError));
          logger.error((0, _util.inspect)(error));
          error = (0, _conversation_complete.createInternalServerError)();
        }
      }
      subscriber.error(error);
    });
  });
}