"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSummarizationFunction = registerSummarizationFunction;
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerSummarizationFunction({
  client,
  functions
}) {
  functions.registerFunction({
    name: 'summarize',
    description: `Use this function to store facts in the knowledge database if the user requests it.
        You can score the learnings with a confidence metric, whether it is a correction on a previous learning.
        An embedding will be created that you can recall later with a semantic search.
        When you create this summarisation, make sure you craft it in a way that can be recalled with a semantic
        search later, and that it would have answered the user's original request.`,
    descriptionForUser: 'This function allows the Elastic Assistant to summarize things from the conversation.',
    parameters: {
      type: 'object',
      properties: {
        id: {
          type: 'string',
          description: 'An id for the document. This should be a short human-readable keyword field with only alphabetic characters and underscores, that allow you to update it later.'
        },
        text: {
          type: 'string',
          description: "A human-readable summary of what you have learned, described in such a way that you can recall it later with semantic search, and that it would have answered the user's original request."
        },
        is_correction: {
          type: 'boolean',
          description: 'Whether this is a correction for a previous learning.'
        },
        confidence: {
          type: 'string',
          description: 'How confident you are about this being a correct and useful learning',
          enum: ['low', 'medium', 'high']
        },
        public: {
          type: 'boolean',
          description: 'Whether this information is specific to the user, or generally applicable to any user of the product'
        }
      },
      required: ['id', 'text', 'is_correction', 'confidence', 'public']
    }
  }, ({
    arguments: {
      id,
      text,
      is_correction: isCorrection,
      confidence,
      public: isPublic
    }
  }, signal) => {
    return client.createKnowledgeBaseEntry({
      entry: {
        doc_id: id,
        role: _common.KnowledgeBaseEntryRole.AssistantSummarization,
        id,
        text,
        is_correction: isCorrection,
        confidence,
        public: isPublic,
        labels: {}
      }
      // signal,
    }).then(() => ({
      content: {
        message: `The document has been stored`
      }
    }));
  });
}