"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGetDatasetInfoFunction = registerGetDatasetInfoFunction;
var _types = require("../../../common/functions/types");
var _get_relevant_field_names = require("./get_relevant_field_names");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerGetDatasetInfoFunction({
  resources,
  functions
}) {
  functions.registerFunction({
    name: 'get_dataset_info',
    visibility: _types.FunctionVisibility.AssistantOnly,
    description: `Use this function to get information about indices/datasets available and the fields available on them.

      providing empty string as index name will retrieve all indices
      else list of all fields for the given index will be given. if no fields are returned this means no indices were matched by provided index pattern.
      wildcards can be part of index name.`,
    descriptionForUser: 'This function allows the assistant to get information about available indices and their fields.',
    parameters: {
      type: 'object',
      additionalProperties: false,
      properties: {
        index: {
          type: 'string',
          description: 'index pattern the user is interested in or empty string to get information about all available indices'
        }
      },
      required: ['index']
    }
  }, async ({
    arguments: {
      index
    },
    messages,
    chat
  }, signal) => {
    const coreContext = await resources.context.core;
    const esClient = coreContext.elasticsearch.client.asCurrentUser;
    const savedObjectsClient = coreContext.savedObjects.client;
    let indices = [];
    try {
      const body = await esClient.indices.resolveIndex({
        name: index === '' ? '*' : index,
        expand_wildcards: 'open'
      });
      indices = [...body.indices.map(i => i.name), ...body.data_streams.map(d => d.name), ...body.aliases.map(d => d.name)];
    } catch (e) {
      indices = [];
    }
    if (index === '') {
      return {
        content: {
          indices,
          fields: []
        }
      };
    }
    if (indices.length === 0) {
      return {
        content: {
          indices,
          fields: []
        }
      };
    }
    const relevantFieldNames = await (0, _get_relevant_field_names.getRelevantFieldNames)({
      index,
      messages,
      esClient,
      dataViews: await resources.plugins.dataViews.start(),
      savedObjectsClient,
      signal,
      chat
    });
    return {
      content: {
        indices: [index],
        fields: relevantFieldNames
      }
    };
  });
}