"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerContextFunction = registerContextFunction;
var _ioTsUtils = require("@kbn/io-ts-utils");
var _dedent = _interopRequireDefault(require("dedent"));
var _gptTokenizer = require("gpt-tokenizer");
var t = _interopRequireWildcard(require("io-ts"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _types = require("../../common/functions/types");
var _types2 = require("../../common/types");
var _concatenate_chat_completion_chunks = require("../../common/utils/concatenate_chat_completion_chunks");
var _create_function_response_message = require("../../common/utils/create_function_response_message");
var _recall_ranking = require("../analytics/recall_ranking");
var _parse_suggestion_scores = require("./parse_suggestion_scores");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_TOKEN_COUNT_FOR_DATA_ON_SCREEN = 1000;
function registerContextFunction({
  client,
  functions,
  resources,
  isKnowledgeBaseAvailable
}) {
  functions.registerFunction({
    name: 'context',
    description: 'This function provides context as to what the user is looking at on their screen, and recalled documents from the knowledge base that matches their query',
    visibility: _types.FunctionVisibility.Internal,
    parameters: {
      type: 'object',
      properties: {
        queries: {
          type: 'array',
          description: 'The query for the semantic search',
          items: {
            type: 'string'
          }
        },
        categories: {
          type: 'array',
          description: 'Categories of internal documentation that you want to search for. By default internal documentation will be excluded. Use `apm` to get internal APM documentation, `lens` to get internal Lens documentation, or both.',
          items: {
            type: 'string',
            enum: ['apm', 'lens']
          }
        }
      },
      required: ['queries', 'categories']
    }
  }, async ({
    arguments: args,
    messages,
    screenContexts,
    chat
  }, signal) => {
    const {
      analytics
    } = (await resources.context.core).coreStart;
    const {
      queries,
      categories
    } = args;
    async function getContext() {
      const screenDescription = (0, _lodash.compact)(screenContexts.map(context => context.screenDescription)).join('\n\n');
      // any data that falls within the token limit, send it automatically

      const dataWithinTokenLimit = (0, _lodash.compact)(screenContexts.flatMap(context => context.data)).filter(data => (0, _gptTokenizer.encode)(JSON.stringify(data.value)).length <= MAX_TOKEN_COUNT_FOR_DATA_ON_SCREEN);
      const content = {
        screen_description: screenDescription,
        learnings: [],
        ...(dataWithinTokenLimit.length ? {
          data_on_screen: dataWithinTokenLimit
        } : {})
      };
      if (!isKnowledgeBaseAvailable) {
        return {
          content
        };
      }
      const userMessage = (0, _lodash.last)(messages.filter(message => message.message.role === _types2.MessageRole.User));
      const nonEmptyQueries = (0, _lodash.compact)(queries);
      const queriesOrUserPrompt = nonEmptyQueries.length ? nonEmptyQueries : (0, _lodash.compact)([userMessage === null || userMessage === void 0 ? void 0 : userMessage.message.content]);
      queriesOrUserPrompt.push(screenDescription);
      const suggestions = await retrieveSuggestions({
        client,
        categories,
        queries: queriesOrUserPrompt
      });
      if (suggestions.length === 0) {
        return {
          content
        };
      }
      try {
        const {
          relevantDocuments,
          scores
        } = await scoreSuggestions({
          suggestions,
          queries: queriesOrUserPrompt,
          messages,
          chat,
          signal,
          logger: resources.logger
        });
        analytics.reportEvent(_recall_ranking.RecallRankingEventType, {
          prompt: queriesOrUserPrompt.join('|'),
          scoredDocuments: suggestions.map(suggestion => {
            var _suggestion$score;
            const llmScore = scores.find(score => score.id === suggestion.id);
            return {
              content: suggestion.text,
              elserScore: (_suggestion$score = suggestion.score) !== null && _suggestion$score !== void 0 ? _suggestion$score : -1,
              llmScore: llmScore ? llmScore.score : -1
            };
          })
        });
        return {
          content: {
            ...content,
            learnings: relevantDocuments
          },
          data: {
            scores,
            suggestions
          }
        };
      } catch (error) {
        return {
          content: {
            ...content,
            learnings: suggestions.slice(0, 5)
          },
          data: {
            error,
            suggestions
          }
        };
      }
    }
    return new _rxjs.Observable(subscriber => {
      getContext().then(({
        content,
        data
      }) => {
        subscriber.next((0, _create_function_response_message.createFunctionResponseMessage)({
          name: 'context',
          content,
          data
        }));
        subscriber.complete();
      }).catch(error => {
        subscriber.error(error);
      });
    });
  });
}
async function retrieveSuggestions({
  queries,
  client,
  categories
}) {
  const recallResponse = await client.recall({
    queries,
    categories
  });
  return recallResponse.entries.map(entry => (0, _lodash.omit)(entry, 'labels', 'is_correction'));
}
const scoreFunctionRequestRt = t.type({
  message: t.type({
    function_call: t.type({
      name: t.literal('score'),
      arguments: t.string
    })
  })
});
const scoreFunctionArgumentsRt = t.type({
  scores: t.string
});
async function scoreSuggestions({
  suggestions,
  messages,
  queries,
  chat,
  signal,
  logger
}) {
  const indexedSuggestions = suggestions.map((suggestion, index) => ({
    ...(0, _lodash.omit)(suggestion, 'score'),
    // To not bias the LLM
    id: index
  }));
  const newUserMessageContent = (0, _dedent.default)(`Given the following question, score the documents that are relevant to the question. on a scale from 0 to 7,
    0 being completely irrelevant, and 7 being extremely relevant. Information is relevant to the question if it helps in
    answering the question. Judge it according to the following criteria:

    - The document is relevant to the question, and the rest of the conversation
    - The document has information relevant to the question that is not mentioned,
      or more detailed than what is available in the conversation
    - The document has a high amount of information relevant to the question compared to other documents
    - The document contains new information not mentioned before in the conversation

    Question:
    ${queries.join('\n')}

    Documents:
    ${JSON.stringify(indexedSuggestions, null, 2)}`);
  const newUserMessage = {
    '@timestamp': new Date().toISOString(),
    message: {
      role: _types2.MessageRole.User,
      content: newUserMessageContent
    }
  };
  const scoreFunction = {
    name: 'score',
    description: 'Use this function to score documents based on how relevant they are to the conversation.',
    parameters: {
      type: 'object',
      properties: {
        scores: {
          description: `The document IDs and their scores, as CSV. Example:
          
            my_id,7
            my_other_id,3
            my_third_id,4
          `,
          type: 'string'
        }
      },
      required: ['score']
    },
    contexts: ['core']
  };
  const response = await (0, _rxjs.lastValueFrom)(chat('score_suggestions', {
    messages: [...messages.slice(0, -2), newUserMessage],
    functions: [scoreFunction],
    functionCall: 'score',
    signal
  }).pipe((0, _concatenate_chat_completion_chunks.concatenateChatCompletionChunks)()));
  const scoreFunctionRequest = (0, _ioTsUtils.decodeOrThrow)(scoreFunctionRequestRt)(response);
  const {
    scores: scoresAsString
  } = (0, _ioTsUtils.decodeOrThrow)(_ioTsUtils.jsonRt.pipe(scoreFunctionArgumentsRt))(scoreFunctionRequest.message.function_call.arguments);
  const scores = (0, _parse_suggestion_scores.parseSuggestionScores)(scoresAsString).map(({
    index,
    score
  }) => {
    return {
      id: suggestions[index].id,
      score
    };
  });
  if (scores.length === 0) {
    // seemingly invalid or no scores, return all
    return {
      relevantDocuments: suggestions,
      scores: []
    };
  }
  const suggestionIds = suggestions.map(document => document.id);
  const relevantDocumentIds = scores.filter(document => suggestionIds.includes(document.id)) // Remove hallucinated documents
  .filter(document => document.score > 4).sort((a, b) => b.score - a.score).slice(0, 5).map(document => document.id);
  const relevantDocuments = suggestions.filter(suggestion => relevantDocumentIds.includes(suggestion.id));
  logger.debug(`Relevant documents: ${JSON.stringify(relevantDocuments, null, 2)}`);
  return {
    relevantDocuments,
    scores
  };
}