"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAlertsFunction = registerAlertsFunction;
var _datemath = _interopRequireDefault(require("@elastic/datemath"));
var _esQuery = require("@kbn/es-query");
var _technical_rule_data_field_names = require("@kbn/rule-registry-plugin/common/technical_rule_data_field_names");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OMITTED_ALERT_FIELDS = ['tags', 'event.action', 'event.kind', 'kibana.alert.rule.execution.uuid', 'kibana.alert.rule.revision', 'kibana.alert.rule.tags', 'kibana.alert.rule.uuid', 'kibana.alert.workflow_status', 'kibana.space_ids', 'kibana.alert.time_range', 'kibana.version'];
const DEFAULT_FEATURE_IDS = ['apm', 'infrastructure', 'logs', 'uptime', 'slo', 'observability'];
function registerAlertsFunction({
  functions,
  resources,
  pluginsStart
}) {
  functions.registerFunction({
    name: 'alerts',
    description: 'Get alerts for Observability. Display the response in tabular format if appropriate.',
    descriptionForUser: 'Get alerts for Observability',
    parameters: {
      type: 'object',
      properties: {
        featureIds: {
          type: 'array',
          items: {
            type: 'string',
            enum: DEFAULT_FEATURE_IDS
          },
          description: 'The Observability apps for which to retrieve alerts. By default it will return alerts for all apps.'
        },
        start: {
          type: 'string',
          description: 'The start of the time range, in Elasticsearch date math, like `now`.'
        },
        end: {
          type: 'string',
          description: 'The end of the time range, in Elasticsearch date math, like `now-24h`.'
        },
        kqlFilter: {
          type: 'string',
          description: 'a KQL query to filter the data by. If no filter should be applied, leave it empty.'
        },
        includeRecovered: {
          type: 'boolean',
          description: 'Whether to include recovered/closed alerts. Defaults to false, which means only active alerts will be returned'
        }
      },
      required: ['start', 'end']
    }
  }, async ({
    arguments: {
      start: startAsDatemath,
      end: endAsDatemath,
      featureIds,
      filter,
      includeRecovered
    }
  }, signal) => {
    const alertsClient = await pluginsStart.ruleRegistry.getRacClientWithRequest(resources.request);
    const start = _datemath.default.parse(startAsDatemath).valueOf();
    const end = _datemath.default.parse(endAsDatemath).valueOf();
    const kqlQuery = !filter ? [] : [(0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(filter))];
    const response = await alertsClient.find({
      featureIds: !!featureIds && !!featureIds.length ? featureIds : DEFAULT_FEATURE_IDS,
      query: {
        bool: {
          filter: [{
            range: {
              '@timestamp': {
                gte: start,
                lte: end
              }
            }
          }, ...kqlQuery, ...(!includeRecovered ? [{
            term: {
              [_technical_rule_data_field_names.ALERT_STATUS]: _technical_rule_data_field_names.ALERT_STATUS_ACTIVE
            }
          }] : [])]
        }
      }
    });

    // trim some fields
    const alerts = response.hits.hits.map(hit => (0, _lodash.omit)(hit._source, ...OMITTED_ALERT_FIELDS));
    return {
      content: {
        total: response.hits.total.value,
        alerts
      }
    };
  });
}