"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getJobDetailsFromTrainedModel = getJobDetailsFromTrainedModel;
exports.getJobsAndModels = getJobsAndModels;
exports.getSavedObjectClientError = getSavedObjectClientError;
exports.mlFunctionsFactory = mlFunctionsFactory;
exports.savedObjectClientsFactory = savedObjectClientsFactory;
var _server = require("@kbn/core/server");
var _saved_objects = require("../../common/types/saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function savedObjectClientsFactory(getSavedObjectsStart) {
  return {
    // create a saved object client scoped to the current request
    // which has access to ml-job objects
    getMlSavedObjectsClient: request => {
      const savedObjectsStart = getSavedObjectsStart();
      if (savedObjectsStart === null) {
        return null;
      }
      return savedObjectsStart.getScopedClient(request, {
        includedHiddenTypes: [_saved_objects.ML_JOB_SAVED_OBJECT_TYPE]
      });
    },
    // create a saved object client which has access to all saved objects
    // no matter the space access of the current user.
    getInternalSavedObjectsClient: () => {
      const savedObjectsStart = getSavedObjectsStart();
      if (savedObjectsStart === null) {
        return null;
      }
      const savedObjectsRepo = savedObjectsStart.createInternalRepository();
      return new _server.SavedObjectsClient(savedObjectsRepo);
    }
  };
}
function getSavedObjectClientError(error) {
  var _error$output, _error$body;
  return error.isBoom && (_error$output = error.output) !== null && _error$output !== void 0 && _error$output.payload ? error.output.payload : (_error$body = error.body) !== null && _error$body !== void 0 ? _error$body : error;
}
function getJobDetailsFromTrainedModel(model) {
  var _model$metadata;
  // @ts-ignore types are wrong
  if (((_model$metadata = model.metadata) === null || _model$metadata === void 0 ? void 0 : _model$metadata.analytics_config) === undefined) {
    return null;
  }

  // @ts-ignore types are wrong
  const jobId = model.metadata.analytics_config.id;
  // @ts-ignore types are wrong
  const createTime = model.metadata.analytics_config.create_time;
  return {
    job_id: jobId,
    create_time: createTime
  };
}

/*
 * Function for calling elasticsearch APIs for retrieving ML jobs and models.
 * The elasticsearch api may be missing in a serverless environment, in which case
 * we return null.
 */

function mlFunctionsFactory(client) {
  return {
    async getJobs() {
      try {
        return await client.asInternalUser.ml.getJobs();
      } catch (error) {
        return null;
      }
    },
    async getDatafeeds() {
      try {
        return await client.asInternalUser.ml.getDatafeeds();
      } catch (error) {
        return null;
      }
    },
    async getTrainedModels() {
      try {
        return await client.asInternalUser.ml.getTrainedModels();
      } catch (error) {
        return null;
      }
    },
    async getDataFrameAnalytics() {
      try {
        return await client.asInternalUser.ml.getDataFrameAnalytics();
      } catch (error) {
        return null;
      }
    }
  };
}

/*
 * Function for retrieving lists of jobs, models and saved objects.
 * If any of the elasticsearch APIs are missing, it returns empty arrays
 * so that the sync process does not create or delete any saved objects.
 */

async function getJobsAndModels(client, mlSavedObjectService) {
  const {
    getJobs,
    getDatafeeds,
    getTrainedModels,
    getDataFrameAnalytics
  } = mlFunctionsFactory(client);
  const [jobObjects, allJobObjects, modelObjects, allModelObjects, adJobs, datafeeds, dfaJobs, models] = await Promise.all([mlSavedObjectService.getAllJobObjects(undefined, false), mlSavedObjectService.getAllJobObjectsForAllSpaces(), mlSavedObjectService.getAllTrainedModelObjects(false), mlSavedObjectService.getAllTrainedModelObjectsForAllSpaces(), getJobs(), getDatafeeds(), getDataFrameAnalytics(), getTrainedModels()]);
  const adJobObjects = adJobs !== null ? jobObjects.filter(j => j.attributes.type === 'anomaly-detector') : [];
  const adAllJobObjects = adJobs !== null ? allJobObjects.filter(j => j.attributes.type === 'anomaly-detector') : [];
  const dfaJobObjects = dfaJobs !== null ? jobObjects.filter(j => j.attributes.type === 'data-frame-analytics') : [];
  const dfaAllJobObjects = dfaJobs !== null ? allJobObjects.filter(j => j.attributes.type === 'data-frame-analytics') : [];
  return {
    jobObjects: [...adJobObjects, ...dfaJobObjects],
    allJobObjects: [...adAllJobObjects, ...dfaAllJobObjects],
    modelObjects: models === null ? [] : modelObjects,
    allModelObjects: models === null ? [] : allModelObjects,
    adJobs: adJobs === null ? [] : adJobs.jobs,
    datafeeds: datafeeds === null ? [] : datafeeds.datafeeds,
    dfaJobs: dfaJobs === null ? [] : dfaJobs.data_frame_analytics,
    models: models === null ? [] : models.trained_model_configs
  };
}