"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useModelMemoryEstimator = exports.modelMemoryEstimatorProvider = void 0;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _new_job = require("../../../../../../../common/constants/new_job");
var _ml_api_service = require("../../../../../services/ml_api_service");
var _job_validator = require("../../job_validator/job_validator");
var _kibana = require("../../../../../contexts/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const modelMemoryEstimatorProvider = (jobCreator, jobValidator) => {
  const modelMemoryCheck$ = new _rxjs.Subject();
  const error$ = new _rxjs.Subject();
  return {
    get error$() {
      return error$.asObservable();
    },
    get updates$() {
      return (0, _rxjs.combineLatest)([jobCreator.wizardInitialized$.pipe((0, _rxjs.skipWhile)(wizardInitialized => wizardInitialized === false)), modelMemoryCheck$]).pipe((0, _rxjs.map)(([, payload]) => payload),
      // delay the request, making sure the validation is completed
      (0, _rxjs.debounceTime)(_job_validator.VALIDATION_DELAY_MS + 100),
      // clone the object to compare payloads and proceed further only
      // if the configuration has been changed
      (0, _rxjs.map)(_lodash.cloneDeep), (0, _rxjs.distinctUntilChanged)(_lodash.isEqual),
      // don't call the endpoint with invalid payload
      (0, _rxjs.filter)(() => jobValidator.isModelMemoryEstimationPayloadValid), (0, _rxjs.switchMap)(payload => {
        return _ml_api_service.ml.calculateModelMemoryLimit$(payload).pipe((0, _rxjs.pluck)('modelMemoryLimit'), (0, _rxjs.catchError)(error => {
          // eslint-disable-next-line no-console
          console.error('Model memory limit could not be calculated', error.body);
          error$.next(error);
          // fallback to the default in case estimation failed
          return (0, _rxjs.of)(_new_job.DEFAULT_MODEL_MEMORY_LIMIT);
        }));
      }));
    },
    update(payload) {
      modelMemoryCheck$.next(payload);
    }
  };
};
exports.modelMemoryEstimatorProvider = modelMemoryEstimatorProvider;
const useModelMemoryEstimator = (jobCreator, jobValidator, jobCreatorUpdate, jobCreatorUpdated) => {
  const {
    services: {
      notifications
    }
  } = (0, _kibana.useMlKibana)();

  // Initialize model memory estimator only once
  const modelMemoryEstimator = (0, _react.useMemo)(() => modelMemoryEstimatorProvider(jobCreator, jobValidator), [jobCreator, jobValidator]);

  // Listen for estimation results and errors
  (0, _react.useEffect)(() => {
    const subscription = new _rxjs.Subscription();
    subscription.add(modelMemoryEstimator.updates$.pipe((0, _rxjs.startWith)(jobCreator.modelMemoryLimit), (0, _rxjs.pairwise)()).subscribe(([previousEstimation, currentEstimation]) => {
      // to make sure we don't overwrite a manual input
      if (jobCreator.modelMemoryLimit === null || jobCreator.modelMemoryLimit === previousEstimation) {
        jobCreator.modelMemoryLimit = currentEstimation;
        // required in order to trigger changes on the input
        jobCreatorUpdate();
      }
    }));
    subscription.add(modelMemoryEstimator.error$.subscribe(error => {
      notifications.toasts.addWarning({
        title: _i18n.i18n.translate('xpack.ml.newJob.wizard.estimateModelMemoryError', {
          defaultMessage: 'Model memory limit could not be calculated'
        }),
        text: _i18n.i18n.translate('xpack.ml.newJob.wizard.estimateModelMemoryErrorText', {
          defaultMessage: '{errorText}. You can proceed with creating the job, but check for warning messages once the job is running that the configured limit has not been exceeded.',
          values: {
            errorText: (0, _mlErrorUtils.extractErrorMessage)(error)
          }
        })
      });
    }));
    return () => {
      subscription.unsubscribe();
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [modelMemoryEstimator]);

  // Update model memory estimation payload on the job creator updates
  (0, _react.useEffect)(() => {
    modelMemoryEstimator.update({
      datafeedConfig: jobCreator.datafeedConfig,
      analysisConfig: jobCreator.jobConfig.analysis_config,
      indexPattern: jobCreator.indexPatternTitle,
      query: jobCreator.datafeedConfig.query,
      timeFieldName: jobCreator.jobConfig.data_description.time_field,
      earliestMs: jobCreator.start,
      latestMs: jobCreator.end
    });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [jobCreator, jobCreatorUpdated]);
};
exports.useModelMemoryEstimator = useModelMemoryEstimator;