"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMapAttributeService = getMapAttributeService;
exports.savedObjectToEmbeddableAttributes = void 0;
var _constants = require("../common/constants");
var _kibana_services = require("./kibana_services");
var _references = require("../common/migrations/references");
var _content_management = require("./content_management");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const savedObjectToEmbeddableAttributes = savedObject => {
  const {
    attributes
  } = (0, _references.injectReferences)(savedObject);
  return {
    ...attributes,
    references: savedObject.references
  };
};
exports.savedObjectToEmbeddableAttributes = savedObjectToEmbeddableAttributes;
let mapAttributeService = null;
function getMapAttributeService() {
  if (mapAttributeService) {
    return mapAttributeService;
  }
  mapAttributeService = (0, _kibana_services.getEmbeddableService)().getAttributeService(_constants.MAP_SAVED_OBJECT_TYPE, {
    saveMethod: async (attributes, savedObjectId) => {
      // AttributeService "attributes" contains "references" as a child.
      // SavedObjectClient "attributes" uses "references" as a sibling.
      // https://github.com/elastic/kibana/issues/83133
      const savedObjectClientReferences = attributes.references;
      const savedObjectClientAttributes = {
        ...attributes
      };
      delete savedObjectClientAttributes.references;
      const {
        attributes: updatedAttributes,
        references
      } = (0, _references.extractReferences)({
        attributes: savedObjectClientAttributes,
        references: savedObjectClientReferences
      });
      const {
        item: {
          id
        }
      } = await (savedObjectId ? (0, _content_management.getMapClient)().update({
        id: savedObjectId,
        data: updatedAttributes,
        options: {
          references
        }
      }) : (0, _content_management.getMapClient)().create({
        data: updatedAttributes,
        options: {
          references
        }
      }));
      return {
        id
      };
    },
    unwrapMethod: async savedObjectId => {
      const {
        item: savedObject,
        meta: {
          outcome,
          aliasPurpose,
          aliasTargetId
        }
      } = await (0, _content_management.getMapClient)().get(savedObjectId);
      if (savedObject.error) {
        throw savedObject.error;
      }
      return {
        attributes: savedObjectToEmbeddableAttributes(savedObject),
        metaInfo: {
          sharingSavedObjectProps: {
            aliasTargetId,
            outcome,
            aliasPurpose,
            sourceId: savedObjectId
          },
          managed: Boolean(savedObject.managed)
        }
      };
    },
    checkForDuplicateTitle: props => {
      return (0, _content_management.checkForDuplicateTitle)({
        title: props.newTitle,
        copyOnSave: false,
        lastSavedTitle: '',
        isTitleDuplicateConfirmed: props.isTitleDuplicateConfirmed,
        getDisplayName: () => _constants.MAP_EMBEDDABLE_NAME,
        onTitleDuplicate: props.onTitleDuplicate
      }, {
        overlays: (0, _kibana_services.getCoreOverlays)()
      });
    }
  });
  return mapAttributeService;
}