"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.colorFromValue = exports.calculateSteppedGradientColor = exports.calculateStepColor = exports.calculateGradientColor = void 0;
var _lodash = require("lodash");
var _polished = require("polished");
var _lib = require("../../../../lib/lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OPERATOR_TO_FN = {
  [_lib.InfraWaffleMapRuleOperator.eq]: _lodash.isEqual,
  [_lib.InfraWaffleMapRuleOperator.lt]: _lodash.lt,
  [_lib.InfraWaffleMapRuleOperator.lte]: _lodash.lte,
  [_lib.InfraWaffleMapRuleOperator.gte]: _lodash.gte,
  [_lib.InfraWaffleMapRuleOperator.gt]: _lodash.gt
};
const convertToRgbString = color => {
  return (0, _polished.toColorString)((0, _polished.parseToRgb)(color));
};
const colorFromValue = (legend, value, bounds, defaultColor = 'rgba(217, 217, 217, 1)') => {
  try {
    if (_lib.StepLegendRT.is(legend)) {
      return convertToRgbString(calculateStepColor(legend.rules, value, defaultColor));
    }
    if (_lib.GradientLegendRT.is(legend)) {
      return convertToRgbString(calculateGradientColor(legend.rules, value, bounds, defaultColor));
    }
    if (_lib.SteppedGradientLegendRT.is(legend)) {
      return convertToRgbString(calculateSteppedGradientColor(legend.rules, value, bounds, defaultColor));
    }
    return defaultColor;
  } catch (error) {
    return defaultColor;
  }
};
exports.colorFromValue = colorFromValue;
const normalizeValue = (min, max, value) => {
  return (value - min) / (max - min);
};
const calculateSteppedGradientColor = (rules, value, bounds, defaultColor = 'rgba(217, 217, 217, 1)') => {
  var _first$color, _first;
  const normalizedValue = normalizeValue(bounds.min, bounds.max, Number(value));
  const steps = rules.length;

  // Since the stepped legend matches a range we need to ensure anything outside
  // the max bounds get's the maximum color.
  const lastRule = (0, _lodash.last)(rules);
  if (lastRule && (0, _lodash.gte)(normalizedValue, lastRule.value)) {
    return lastRule.color;
  }
  return rules.reduce((color, rule) => {
    const min = rule.value - 1 / steps;
    const max = rule.value;
    if ((0, _lodash.gte)(normalizedValue, min) && (0, _lodash.lte)(normalizedValue, max)) {
      return rule.color;
    }
    return color;
  }, (_first$color = (_first = (0, _lodash.first)(rules)) === null || _first === void 0 ? void 0 : _first.color) !== null && _first$color !== void 0 ? _first$color : defaultColor);
};
exports.calculateSteppedGradientColor = calculateSteppedGradientColor;
const calculateStepColor = (rules, value, defaultColor = 'rgba(217, 217, 217, 1)') => {
  return rules.reduce((color, rule) => {
    const operatorFn = OPERATOR_TO_FN[rule.operator];
    if (operatorFn(value, rule.value)) {
      return rule.color;
    }
    return color;
  }, defaultColor);
};
exports.calculateStepColor = calculateStepColor;
const calculateGradientColor = (rules, value, bounds, defaultColor = 'rgba(0, 179, 164, 1)') => {
  if (rules.length === 0) {
    return defaultColor;
  }
  if (rules.length === 1) {
    return (0, _lodash.last)(rules).color;
  }
  const {
    min,
    max
  } = bounds;
  const sortedRules = (0, _lodash.sortBy)(rules, 'value');
  const normValue = normalizeValue(min, max, Number(value));
  const startRule = sortedRules.reduce((acc, rule) => {
    if (rule.value <= normValue) {
      return rule;
    }
    return acc;
  }, (0, _lodash.first)(sortedRules));
  const endRule = sortedRules.filter(r => r !== startRule).find(r => r.value >= normValue);
  if (!endRule) {
    return startRule.color;
  }
  const mixValue = normalizeValue(startRule.value, endRule.value, normValue);
  return (0, _polished.mix)(mixValue, endRule.color, startRule.color);
};
exports.calculateGradientColor = calculateGradientColor;