"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateProperties = exports.validateMappingsConfiguration = exports.validateMappings = exports.mappingsConfigurationSchemaKeys = exports.mappingsConfigurationSchema = void 0;
var _lodash = require("lodash");
var t = _interopRequireWildcard(require("io-ts"));
var _Ord = require("fp-ts/lib/Ord");
var _Set = require("fp-ts/lib/Set");
var _Either = require("fp-ts/lib/Either");
var _error_reporter = require("./error_reporter");
var _constants = require("../constants");
var _utils = require("./utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ALLOWED_FIELD_PROPERTIES = [...Object.keys(_constants.PARAMETERS_DEFINITION), 'type', 'properties', 'fields'];
const DEFAULT_FIELD_TYPE = 'object';
const validateFieldType = type => {
  if (typeof type !== 'string') {
    return false;
  }
  if (!_constants.ALL_DATA_TYPES.includes(type)) {
    return false;
  }
  return true;
};
const validateParameter = (parameter, value) => {
  if (parameter === 'type') {
    return true;
  }
  if (parameter === 'name') {
    return false;
  }
  if (parameter === 'properties' || parameter === 'fields') {
    return (0, _lodash.isPlainObject)(value);
  }
  const parameterSchema = _constants.PARAMETERS_DEFINITION[parameter].schema;
  if (parameterSchema) {
    return (0, _Either.isRight)(parameterSchema.decode(value));
  }

  // Fallback, if no schema defined for the parameter (this should not happen in theory)
  return true;
};
const stripUnknownOrInvalidParameter = field => Object.entries(field).reduce((acc, [key, value]) => {
  if (!ALLOWED_FIELD_PROPERTIES.includes(key) || !validateParameter(key, value)) {
    acc.parametersRemoved.push(key);
  } else {
    var _acc$value;
    acc.value = (_acc$value = acc.value) !== null && _acc$value !== void 0 ? _acc$value : {};
    acc.value[key] = value;
  }
  return acc;
}, {
  parametersRemoved: []
});
const parseField = field => {
  var _field$type;
  // Sanitize the input to make sure we are working with an object
  if (!(0, _lodash.isPlainObject)(field)) {
    return {
      parametersRemoved: []
    };
  }
  // Make sure the field "type" is valid
  if (!validateFieldType((_field$type = field.type) !== null && _field$type !== void 0 ? _field$type : DEFAULT_FIELD_TYPE)) {
    return {
      parametersRemoved: []
    };
  }

  // Filter out unknown or invalid "parameters"
  const fieldWithType = {
    type: DEFAULT_FIELD_TYPE,
    ...field
  };
  const parsedField = stripUnknownOrInvalidParameter(fieldWithType);
  const meta = (0, _utils.getFieldMeta)(fieldWithType);
  return {
    ...parsedField,
    meta
  };
};
const parseFields = (properties, path = []) => {
  return Object.entries(properties).reduce((acc, [fieldName, unparsedField]) => {
    const fieldPath = [...path, fieldName].join('.');
    const {
      value: parsedField,
      parametersRemoved,
      meta
    } = parseField(unparsedField);
    if (parsedField === undefined) {
      // Field has been stripped out because it was invalid
      acc.errors.push({
        code: 'ERR_FIELD',
        fieldPath
      });
    } else {
      if (meta.hasChildFields || meta.hasMultiFields) {
        // Recursively parse all the possible children ("properties" or "fields" for multi-fields)
        const parsedChildren = parseFields(parsedField[meta.childFieldsName], [...path, fieldName]);
        parsedField[meta.childFieldsName] = parsedChildren.value;

        /**
         * If the children parsed have any error we concatenate them in our accumulator.
         */
        if (parsedChildren.errors) {
          acc.errors = [...acc.errors, ...parsedChildren.errors];
        }
      }
      acc.value[fieldName] = parsedField;
      if (Boolean(parametersRemoved.length)) {
        acc.errors = [...acc.errors, ...parametersRemoved.map(paramName => ({
          code: 'ERR_PARAMETER',
          fieldPath,
          paramName
        }))];
      }
    }
    return acc;
  }, {
    value: {},
    errors: []
  });
};

/**
 * Utility function that reads a mappings "properties" object and validate its fields by
 * - Removing unknown field types
 * - Removing unknown field parameters or field parameters that don't have the correct format.
 *
 * This method does not mutate the original properties object. It returns an object with
 * the parsed properties and an array of field paths that have been removed.
 * This allows us to display a warning in the UI and let the user correct the fields that we
 * are about to remove.
 *
 * NOTE: The Joi Schema that we defined for each parameter (in "parameters_definition".tsx)
 * does not do an exhaustive validation of the parameter value.
 * It's main purpose is to prevent the UI from blowing up.
 *
 * @param properties A mappings "properties" object
 */
const validateProperties = (properties = {}) => {
  // Sanitize the input to make sure we are working with an object
  if (!(0, _lodash.isPlainObject)(properties)) {
    return {
      value: {},
      errors: []
    };
  }
  return parseFields(properties);
};

/**
 * Single source of truth to validate the *configuration* of the mappings.
 * Whenever a user loads a JSON object it will be validate against this Joi schema.
 */
exports.validateProperties = validateProperties;
const mappingsConfigurationSchema = exports.mappingsConfigurationSchema = t.exact(t.partial({
  properties: t.UnknownRecord,
  _data_stream_timestamp: t.interface({
    enabled: t.boolean
  }),
  runtime: t.UnknownRecord,
  dynamic: t.union([t.literal(true), t.literal(false), t.literal('strict'), t.literal('true'), t.literal('false'), t.literal('runtime')]),
  date_detection: t.boolean,
  numeric_detection: t.boolean,
  dynamic_date_formats: t.array(t.string),
  _source: t.exact(t.partial({
    enabled: t.boolean,
    includes: t.array(t.string),
    excludes: t.array(t.string)
  })),
  _meta: t.UnknownRecord,
  _routing: t.interface({
    required: t.boolean
  }),
  dynamic_templates: t.array(t.UnknownRecord),
  // Mapper size plugin
  _size: t.interface({
    enabled: t.boolean
  })
}));
const mappingsConfigurationSchemaKeys = exports.mappingsConfigurationSchemaKeys = Object.keys(mappingsConfigurationSchema.type.props);
const sourceConfigurationSchemaKeys = Object.keys(mappingsConfigurationSchema.type.props._source.type.props);
const validateMappingsConfiguration = mappingsConfiguration => {
  // Set to keep track of invalid configuration parameters.
  const configurationRemoved = new Set();
  let copyOfMappingsConfig = {
    ...mappingsConfiguration
  };
  const result = mappingsConfigurationSchema.decode(mappingsConfiguration);
  const isSchemaInvalid = (0, _Either.isLeft)(result);
  const unknownConfigurationParameters = Object.keys(mappingsConfiguration).filter(key => mappingsConfigurationSchemaKeys.includes(key) === false);
  const unknownSourceConfigurationParameters = mappingsConfiguration._source !== undefined ? Object.keys(mappingsConfiguration._source).filter(key => sourceConfigurationSchemaKeys.includes(key) === false) : [];
  if (isSchemaInvalid) {
    /**
     * To keep the logic simple we will strip out the parameters that contain errors
     */
    const errors = _error_reporter.errorReporter.report(result);
    errors.forEach(error => {
      const configurationName = error.path[0];
      configurationRemoved.add(configurationName);
      delete copyOfMappingsConfig[configurationName];
    });
  }
  if (unknownConfigurationParameters.length > 0) {
    unknownConfigurationParameters.forEach(configName => configurationRemoved.add(configName));
  }
  if (unknownSourceConfigurationParameters.length > 0) {
    configurationRemoved.add('_source');
    delete copyOfMappingsConfig._source;
  }
  copyOfMappingsConfig = (0, _lodash.pick)(copyOfMappingsConfig, mappingsConfigurationSchemaKeys);
  const errors = (0, _Set.toArray)(_Ord.ordString)(configurationRemoved).map(configName => ({
    code: 'ERR_CONFIG',
    configName
  })).sort((a, b) => a.configName.localeCompare(b.configName));
  return {
    value: copyOfMappingsConfig,
    errors
  };
};
exports.validateMappingsConfiguration = validateMappingsConfiguration;
const validatePluginsParameters = esNodesPlugins => mappingsConfiguration => {
  const copyOfMappingsConfig = {
    ...mappingsConfiguration
  };
  const errors = [];

  // Mapper size plugin parameters
  if ('_size' in copyOfMappingsConfig && !esNodesPlugins.includes(_constants.MapperSizePluginId)) {
    errors.push({
      code: 'ERR_CONFIG',
      configName: '_size'
    });
    delete copyOfMappingsConfig._size;
  }
  return {
    value: copyOfMappingsConfig,
    errors
  };
};
const validateMappings = (mappings = {}, esNodesPlugins = []) => {
  if (!(0, _lodash.isPlainObject)(mappings)) {
    return {
      value: {}
    };
  }
  const {
    properties,
    dynamic_templates: dynamicTemplates,
    ...mappingsConfiguration // extract the mappings configuration
  } = mappings;

  // Run the different validators on the mappings configuration. Each validator returns
  // the mapping configuration sanitized (in "value") and possible errors found.
  const {
    value: parsedConfiguration,
    errors: configurationErrors
  } = [validateMappingsConfiguration, validatePluginsParameters(esNodesPlugins)].reduce((acc, validator) => {
    const {
      value: sanitizedConfiguration,
      errors: validationErrors
    } = validator(acc.value);
    return {
      value: sanitizedConfiguration,
      errors: [...acc.errors, ...validationErrors]
    };
  }, {
    value: mappingsConfiguration,
    errors: []
  });
  const {
    value: parsedProperties,
    errors: propertiesErrors
  } = validateProperties(properties);
  const errors = [...configurationErrors, ...propertiesErrors];
  return {
    value: {
      ...parsedConfiguration,
      properties: parsedProperties,
      dynamic_templates: dynamicTemplates !== null && dynamicTemplates !== void 0 ? dynamicTemplates : []
    },
    errors: errors.length ? errors : undefined
  };
};
exports.validateMappings = validateMappings;