"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._installPackage = _installPackage;
var _server = require("@kbn/core/server");
var _services = require("../../../../common/services");
var _constants = require("../../../../common/constants");
var _constants2 = require("../../../constants");
var _remove_legacy = require("../elasticsearch/template/remove_legacy");
var _ingest_pipeline = require("../elasticsearch/ingest_pipeline");
var _install = require("../elasticsearch/ilm/install");
var _install2 = require("../kibana/assets/install");
var _template = require("../elasticsearch/template/template");
var _install3 = require("../elasticsearch/transform/install");
var _ml_model = require("../elasticsearch/ml_model");
var _install4 = require("../elasticsearch/datastream_ilm/install");
var _storage = require("../archive/storage");
var _errors = require("../../../errors");
var _ = require("../..");
var _audit_logging = require("../../audit_logging");
var _install5 = require("./install");
var _utils = require("./utils");
var _install_errors_helpers = require("./install_errors_helpers");
var _install_index_template_pipeline = require("./install_index_template_pipeline");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// this is only exported for testing
// use a leading underscore to indicate it's not the supported path
// only the more explicit `installPackage*` functions should be used
async function _installPackage({
  savedObjectsClient,
  savedObjectsImporter,
  savedObjectTagAssignmentService,
  savedObjectTagClient,
  esClient,
  logger,
  installedPkg,
  packageInstallContext,
  installType,
  installSource,
  spaceId,
  force,
  verificationResult,
  authorizationHeader,
  ignoreMappingUpdateErrors,
  skipDataStreamRollover
}) {
  const {
    packageInfo,
    paths
  } = packageInstallContext;
  const {
    name: pkgName,
    version: pkgVersion,
    title: pkgTitle
  } = packageInfo;
  try {
    var _installedPkg$attribu, _appContextService$ge, _appContextService$ge2, _appContextService$ge3, _updatedPackage$attri;
    // if some installation already exists
    if (installedPkg) {
      const isStatusInstalling = installedPkg.attributes.install_status === 'installing';
      const hasExceededTimeout = Date.now() - Date.parse(installedPkg.attributes.install_started_at) < _constants.MAX_TIME_COMPLETE_INSTALL;
      logger.debug(`Package install - Install status ${pkgName}-${pkgVersion}: ${installedPkg.attributes.install_status}`);

      // if the installation is currently running, don't try to install
      // instead, only return already installed assets
      if (isStatusInstalling && hasExceededTimeout) {
        // If this is a forced installation, ignore the timeout and restart the installation anyway
        logger.debug(`Package install - Installation is running and has exceeded timeout`);
        if (force) {
          logger.debug(`Package install - Forced installation, restarting`);
          await (0, _install5.restartInstallation)({
            savedObjectsClient,
            pkgName,
            pkgVersion,
            installSource,
            verificationResult
          });
        } else {
          throw new _errors.ConcurrentInstallOperationError(`Concurrent installation or upgrade of ${pkgName || 'unknown'}-${pkgVersion || 'unknown'} detected, aborting.`);
        }
      } else {
        // if no installation is running, or the installation has been running longer than MAX_TIME_COMPLETE_INSTALL
        // (it might be stuck) update the saved object and proceed
        logger.debug(`Package install - no installation running or the installation has been running longer than ${_constants.MAX_TIME_COMPLETE_INSTALL}, restarting`);
        await (0, _install5.restartInstallation)({
          savedObjectsClient,
          pkgName,
          pkgVersion,
          installSource,
          verificationResult
        });
      }
    } else {
      logger.debug(`Package install - Create installation ${pkgName}-${pkgVersion}`);
      await (0, _install5.createInstallation)({
        savedObjectsClient,
        packageInfo,
        installSource,
        spaceId,
        verificationResult
      });
    }
    logger.debug(`Package install - Installing Kibana assets`);
    const kibanaAssetPromise = (0, _utils.withPackageSpan)('Install Kibana assets', () => (0, _install2.installKibanaAssetsAndReferences)({
      savedObjectsClient,
      savedObjectsImporter,
      savedObjectTagAssignmentService,
      savedObjectTagClient,
      pkgName,
      pkgTitle,
      packageInstallContext,
      installedPkg,
      logger,
      spaceId,
      assetTags: packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.asset_tags
    }));
    // Necessary to avoid async promise rejection warning
    // See https://stackoverflow.com/questions/40920179/should-i-refrain-from-handling-promise-rejection-asynchronously
    kibanaAssetPromise.catch(() => {});

    // Use a shared array that is updated by each operation. This allows each operation to accurately update the
    // installation object with it's references without requiring a refresh of the SO index on each update (faster).
    let esReferences = (_installedPkg$attribu = installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.installed_es) !== null && _installedPkg$attribu !== void 0 ? _installedPkg$attribu : [];

    // the rest of the installation must happen in sequential order
    // currently only the base package has an ILM policy
    // at some point ILM policies can be installed/modified
    // per data stream and we should then save them
    const isILMPoliciesDisabled = (_appContextService$ge = (_appContextService$ge2 = _.appContextService.getConfig()) === null || _appContextService$ge2 === void 0 ? void 0 : (_appContextService$ge3 = _appContextService$ge2.internal) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.disableILMPolicies) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : false;
    if (!isILMPoliciesDisabled) {
      esReferences = await (0, _utils.withPackageSpan)('Install ILM policies', () => (0, _install.installILMPolicy)(packageInstallContext, esClient, savedObjectsClient, logger, esReferences));
      logger.debug(`Package install - Installing Data Stream ILM policies`);
      ({
        esReferences
      } = await (0, _utils.withPackageSpan)('Install Data Stream ILM policies', () => (0, _install4.installIlmForDataStream)(packageInstallContext, esClient, savedObjectsClient, logger, esReferences)));
    }

    // installs ml models
    logger.debug(`Package install - installing ML models`);
    esReferences = await (0, _utils.withPackageSpan)('Install ML models', () => (0, _ml_model.installMlModel)(packageInstallContext, esClient, savedObjectsClient, logger, esReferences));
    let indexTemplates = [];
    if (packageInfo.type === 'integration') {
      logger.debug(`Package install - Installing index templates and pipelines, packageInfo.type ${packageInfo.type}`);
      const {
        installedTemplates,
        esReferences: templateEsReferences
      } = await (0, _install_index_template_pipeline.installIndexTemplatesAndPipelines)({
        installedPkg: installedPkg ? installedPkg.attributes : undefined,
        packageInstallContext,
        esClient,
        savedObjectsClient,
        logger,
        esReferences
      });
      esReferences = templateEsReferences;
      indexTemplates = installedTemplates;
    }
    if (packageInfo.type === 'input' && installedPkg) {
      // input packages create their data streams during package policy creation
      // we must use installed_es to infer which streams exist first then
      // we can install the new index templates
      logger.debug(`Package install - packageInfo.type ${packageInfo.type}`);
      const dataStreamNames = installedPkg.attributes.installed_es.filter(ref => ref.type === 'index_template')
      // index templates are named {type}-{dataset}, remove everything before first hyphen
      .map(ref => ref.id.replace(/^[^-]+-/, ''));
      const dataStreams = dataStreamNames.flatMap(dataStreamName => (0, _services.getNormalizedDataStreams)(packageInfo, dataStreamName));
      if (dataStreams.length) {
        logger.debug(`Package install - installing index templates and pipelines with datastreams length ${dataStreams.length}`);
        const {
          installedTemplates,
          esReferences: templateEsReferences
        } = await (0, _install_index_template_pipeline.installIndexTemplatesAndPipelines)({
          installedPkg: installedPkg ? installedPkg.attributes : undefined,
          packageInstallContext,
          esClient,
          savedObjectsClient,
          logger,
          esReferences,
          onlyForDataStreams: dataStreams
        });
        esReferences = templateEsReferences;
        indexTemplates = installedTemplates;
      }
    }
    try {
      logger.debug(`Package install - Removing legacy templates`);
      await (0, _remove_legacy.removeLegacyTemplates)({
        packageInfo,
        esClient,
        logger
      });
    } catch (e) {
      logger.warn(`Error removing legacy templates: ${e.message}`);
    }

    // update current backing indices of each data stream
    logger.debug(`Package install - Updating backing indices of each data stream`);
    await (0, _utils.withPackageSpan)('Update write indices', () => (0, _template.updateCurrentWriteIndices)(esClient, logger, indexTemplates, {
      ignoreMappingUpdateErrors,
      skipDataStreamRollover
    }));
    logger.debug(`Package install - Installing transforms`);
    ({
      esReferences
    } = await (0, _utils.withPackageSpan)('Install transforms', () => (0, _install3.installTransforms)({
      packageInstallContext,
      esClient,
      savedObjectsClient,
      logger,
      esReferences,
      force,
      authorizationHeader
    })));

    // If this is an update or retrying an update, delete the previous version's pipelines
    // Top-level pipeline assets will not be removed on upgrade as of ml model package addition which requires previous
    // assets to remain installed. This is a temporary solution - more robust solution tracked here https://github.com/elastic/kibana/issues/115035
    if (paths.filter(path => (0, _ingest_pipeline.isTopLevelPipeline)(path)).length === 0 && (installType === 'update' || installType === 'reupdate') && installedPkg) {
      logger.debug(`Package install - installType ${installType} Deleting previous ingest pipelines`);
      esReferences = await (0, _utils.withPackageSpan)('Delete previous ingest pipelines', () => (0, _ingest_pipeline.deletePreviousPipelines)(esClient, savedObjectsClient, pkgName, installedPkg.attributes.version, esReferences));
    }
    // pipelines from a different version may have installed during a failed update
    if (installType === 'rollback' && installedPkg) {
      logger.debug(`Package install - installType ${installType} Deleting previous ingest pipelines`);
      esReferences = await (0, _utils.withPackageSpan)('Delete previous ingest pipelines', () => (0, _ingest_pipeline.deletePreviousPipelines)(esClient, savedObjectsClient, pkgName, installedPkg.attributes.install_version, esReferences));
    }
    const installedKibanaAssetsRefs = await kibanaAssetPromise;
    logger.debug(`Package install - Updating archive entries`);
    const packageAssetResults = await (0, _utils.withPackageSpan)('Update archive entries', () => (0, _storage.saveArchiveEntriesFromAssetsMap)({
      savedObjectsClient,
      assetsMap: packageInstallContext.assetsMap,
      paths: packageInstallContext.paths,
      packageInfo,
      installSource
    }));
    const packageAssetRefs = packageAssetResults.saved_objects.map(result => ({
      id: result.id,
      type: _constants.ASSETS_SAVED_OBJECT_TYPE
    }));
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id: pkgName,
      savedObjectType: _constants2.PACKAGES_SAVED_OBJECT_TYPE
    });
    logger.debug(`Package install - Updating install status`);
    await (0, _utils.withPackageSpan)('Update install status', () => {
      var _installedPkg$attribu2;
      return savedObjectsClient.update(_constants2.PACKAGES_SAVED_OBJECT_TYPE, pkgName, {
        version: pkgVersion,
        install_version: pkgVersion,
        install_status: 'installed',
        package_assets: packageAssetRefs,
        install_format_schema_version: _constants2.FLEET_INSTALL_FORMAT_VERSION,
        latest_install_failed_attempts: (0, _install_errors_helpers.clearLatestFailedAttempts)(pkgVersion, (_installedPkg$attribu2 = installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.latest_install_failed_attempts) !== null && _installedPkg$attribu2 !== void 0 ? _installedPkg$attribu2 : [])
      });
    });

    // Need to refetch the installation again to retrieve all the attributes
    const updatedPackage = await savedObjectsClient.get(_constants2.PACKAGES_SAVED_OBJECT_TYPE, pkgName);
    logger.debug(`Package install - Install status ${updatedPackage === null || updatedPackage === void 0 ? void 0 : (_updatedPackage$attri = updatedPackage.attributes) === null || _updatedPackage$attri === void 0 ? void 0 : _updatedPackage$attri.install_status}`);
    // If the package is flagged with the `keep_policies_up_to_date` flag, upgrade its
    // associated package policies after installation
    if (updatedPackage.attributes.keep_policies_up_to_date) {
      await (0, _utils.withPackageSpan)('Upgrade package policies', async () => {
        const policyIdsToUpgrade = await _.packagePolicyService.listIds(savedObjectsClient, {
          page: 1,
          perPage: _constants.SO_SEARCH_LIMIT,
          kuery: `${_constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${pkgName}`
        });
        logger.debug(`Package install - Package is flagged with keep_policies_up_to_date, upgrading its associated package policies ${policyIdsToUpgrade}`);
        await _.packagePolicyService.upgrade(savedObjectsClient, esClient, policyIdsToUpgrade.items);
      });
    }
    logger.debug(`Package install - Installation complete`);
    return [...installedKibanaAssetsRefs, ...esReferences];
  } catch (err) {
    if (_server.SavedObjectsErrorHelpers.isConflictError(err)) {
      throw new _errors.PackageSavedObjectConflictError(`Saved Object conflict encountered while installing ${pkgName || 'unknown'}-${pkgVersion || 'unknown'}. There may be a conflicting Saved Object saved to another Space. Original error: ${err.message}`);
    } else {
      throw err;
    }
  }
}