"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addCustomPipelineAndLocalRoutingRulesProcessor = addCustomPipelineAndLocalRoutingRulesProcessor;
exports.isTopLevelPipeline = exports.getPipelineNameForInstallation = void 0;
exports.rewriteIngestPipeline = rewriteIngestPipeline;
var _jsYaml = require("js-yaml");
var _types = require("../../../../types");
var _archive = require("../../archive");
var _services = require("../../../../../common/services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isTopLevelPipeline = path => {
  const pathParts = (0, _archive.getPathParts)(path);
  return pathParts.type === _types.ElasticsearchAssetType.ingestPipeline && pathParts.dataset === undefined;
};
exports.isTopLevelPipeline = isTopLevelPipeline;
const getPipelineNameForInstallation = ({
  pipelineName,
  dataStream,
  packageVersion
}) => {
  if (dataStream !== undefined) {
    const isPipelineEntry = pipelineName === dataStream.ingest_pipeline;
    const suffix = isPipelineEntry ? '' : `-${pipelineName}`;
    // if this is the pipeline entry, don't add a suffix
    return `${(0, _services.getPipelineNameForDatastream)({
      dataStream,
      packageVersion
    })}${suffix}`;
  }
  // It's a top-level pipeline
  return `${packageVersion}-${pipelineName}`;
};
exports.getPipelineNameForInstallation = getPipelineNameForInstallation;
function rewriteIngestPipeline(pipeline, substitutions) {
  substitutions.forEach(sub => {
    const {
      source,
      target,
      templateFunction
    } = sub;
    // This fakes the use of the golang text/template expression {{SomeTemplateFunction 'some-param'}}
    // cf. https://github.com/elastic/beats/blob/master/filebeat/fileset/fileset.go#L294

    // "Standard style" uses '{{' and '}}' as delimiters
    const matchStandardStyle = `{{\\s?${templateFunction}\\s+['"]${source}['"]\\s?}}`;
    // "Beats style" uses '{<' and '>}' as delimiters because this is current practice in the beats project
    const matchBeatsStyle = `{<\\s?${templateFunction}\\s+['"]${source}['"]\\s?>}`;
    const regexStandardStyle = new RegExp(matchStandardStyle);
    const regexBeatsStyle = new RegExp(matchBeatsStyle);
    pipeline = pipeline.replace(regexStandardStyle, target).replace(regexBeatsStyle, target);
  });
  return pipeline;
}
function mutatePipelineContentWithNewProcessor(jsonPipelineContent, processor) {
  if (!jsonPipelineContent.processors) {
    jsonPipelineContent.processors = [];
  }
  jsonPipelineContent.processors.push(processor);
}
function addCustomPipelineAndLocalRoutingRulesProcessor(pipeline) {
  var _pipeline$dataStream$, _pipeline$dataStream, _pipeline$dataStream$2, _pipeline$dataStream$3;
  if (!pipeline.shouldInstallCustomPipelines || !pipeline.dataStream) {
    return pipeline;
  }
  const localRoutingRules = (_pipeline$dataStream$ = (_pipeline$dataStream = pipeline.dataStream) === null || _pipeline$dataStream === void 0 ? void 0 : (_pipeline$dataStream$2 = _pipeline$dataStream.routing_rules) === null || _pipeline$dataStream$2 === void 0 ? void 0 : (_pipeline$dataStream$3 = _pipeline$dataStream$2.find(rule => {
    var _pipeline$dataStream2;
    return rule.source_dataset === ((_pipeline$dataStream2 = pipeline.dataStream) === null || _pipeline$dataStream2 === void 0 ? void 0 : _pipeline$dataStream2.dataset);
  })) === null || _pipeline$dataStream$3 === void 0 ? void 0 : _pipeline$dataStream$3.rules) !== null && _pipeline$dataStream$ !== void 0 ? _pipeline$dataStream$ : [];
  const customPipelineProcessors = [{
    pipeline: {
      name: 'global@custom',
      ignore_missing_pipeline: true,
      description: '[Fleet] Global pipeline for all data streams'
    }
  }, {
    pipeline: {
      name: `${pipeline.dataStream.type}@custom`,
      ignore_missing_pipeline: true,
      description: `[Fleet] Pipeline for all data streams of type \`${pipeline.dataStream.type}\``
    }
  }, ...(pipeline.dataStream.package ? [{
    pipeline: {
      // This pipeline name gets the `.integration` suffix to avoid conflicts with the pipeline name for the dataset below
      name: `${pipeline.dataStream.type}-${pipeline.dataStream.package}.integration@custom`,
      ignore_missing_pipeline: true,
      description: `[Fleet] Pipeline for all data streams of type \`${pipeline.dataStream.type}\` defined by the \`${pipeline.dataStream.package}\` integration`
    }
  }] : []), {
    pipeline: {
      name: `${pipeline.dataStream.type}-${pipeline.dataStream.dataset}@custom`,
      ignore_missing_pipeline: true,
      description: `[Fleet] Pipeline for the \`${pipeline.dataStream.dataset}\` dataset`
    }
  }];
  const rerouteProcessors = localRoutingRules.map(routingRule => {
    var _pipeline$dataStream3;
    return {
      reroute: {
        tag: (_pipeline$dataStream3 = pipeline.dataStream) === null || _pipeline$dataStream3 === void 0 ? void 0 : _pipeline$dataStream3.dataset,
        dataset: routingRule.target_dataset,
        namespace: routingRule.namespace,
        if: routingRule.if
      }
    };
  });
  if (pipeline.extension === 'yml') {
    const parsedPipelineContent = (0, _jsYaml.safeLoad)(pipeline.contentForInstallation);
    customPipelineProcessors.forEach(processor => mutatePipelineContentWithNewProcessor(parsedPipelineContent, processor));
    rerouteProcessors.forEach(processor => mutatePipelineContentWithNewProcessor(parsedPipelineContent, processor));
    return {
      ...pipeline,
      contentForInstallation: `---\n${(0, _jsYaml.safeDump)(parsedPipelineContent)}`
    };
  }
  const parsedPipelineContent = JSON.parse(pipeline.contentForInstallation);
  customPipelineProcessors.forEach(processor => mutatePipelineContentWithNewProcessor(parsedPipelineContent, processor));
  rerouteProcessors.forEach(processor => mutatePipelineContentWithNewProcessor(parsedPipelineContent, processor));
  return {
    ...pipeline,
    contentForInstallation: JSON.stringify(parsedPipelineContent)
  };
}