"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PolicyWatcher = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _common = require("../../common");
var _agent_policy_config = require("../../common/services/agent_policy_config");
var _agent_policy = require("./agent_policy");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class PolicyWatcher {
  constructor(soStart, logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "subscription", void 0);
    (0, _defineProperty2.default)(this, "soStart", void 0);
    this.logger = logger;
    this.soStart = soStart;
  }

  /**
   * The policy watcher is not called as part of a HTTP request chain, where the
   * request-scoped SOClient could be passed down. It is called via license observable
   * changes. We are acting as the 'system' in response to license changes, so we are
   * intentionally using the system user here. Be very aware of what you are using this
   * client to do
   */
  makeInternalSOClient(soStart) {
    const fakeRequest = {
      headers: {},
      getBasePath: () => '',
      path: '/',
      route: {
        settings: {}
      },
      url: {
        href: {}
      },
      raw: {
        req: {
          url: '/'
        }
      }
    };
    return soStart.getScopedClient(fakeRequest, {
      excludedExtensions: [_coreSavedObjectsServer.SECURITY_EXTENSION_ID]
    });
  }
  start(licenseService) {
    var _licenseService$getLi;
    this.subscription = (_licenseService$getLi = licenseService.getLicenseInformation$()) === null || _licenseService$getLi === void 0 ? void 0 : _licenseService$getLi.subscribe(this.watch.bind(this));
  }
  stop() {
    if (this.subscription) {
      this.subscription.unsubscribe();
    }
  }
  async watch(license) {
    const log = this.logger.get('endpoint', 'agentPolicyLicenseWatch');
    const agentPolicyFetcher = _agent_policy.agentPolicyService.fetchAllAgentPolicies(this.makeInternalSOClient(this.soStart), {
      fields: ['is_protected', 'id', 'revision']
    } // Don't forget to extend this to include all fields that are used in the `isAgentPolicyValidForLicense` function
    );
    log.info('Checking agent policies for compliance with the current license.');
    const updatedAgentPolicies = [];
    for await (const agentPolicyPageResults of agentPolicyFetcher) {
      const policiesToUpdate = agentPolicyPageResults.reduce((acc, policy) => {
        if (!(0, _agent_policy_config.isAgentPolicyValidForLicense)(policy, license)) {
          acc.push((0, _agent_policy_config.unsetAgentPolicyAccordingToLicenseLevel)(policy, license));
        }
        return acc;
      }, []);
      if (policiesToUpdate.length === 0) {
        break;
      }
      const {
        saved_objects: bulkUpdateSavedObjects
      } = await this.makeInternalSOClient(this.soStart).bulkUpdate(policiesToUpdate.map(policy => {
        const {
          id,
          revision,
          ...policyContent
        } = policy;
        return {
          type: _common.AGENT_POLICY_SAVED_OBJECT_TYPE,
          id,
          attributes: {
            ...policyContent,
            revision: revision + 1,
            updated_at: new Date().toISOString(),
            updated_by: 'system'
          }
        };
      }));
      updatedAgentPolicies.push(...bulkUpdateSavedObjects);
    }
    const failedPolicies = [];
    updatedAgentPolicies.forEach(policy => {
      if (policy.error) {
        failedPolicies.push({
          id: policy.id,
          error: policy.error
        });
      }
    });
    const updatedPoliciesSuccess = updatedAgentPolicies.filter(policy => !policy.error);
    if (!updatedPoliciesSuccess.length && !failedPolicies.length) {
      log.info(`All agent policies are compliant, nothing to do!`);
    } else if (updatedPoliciesSuccess.length && failedPolicies.length) {
      const totalPolicies = updatedPoliciesSuccess.length + failedPolicies.length;
      log.error(`Done - ${failedPolicies.length} out of ${totalPolicies} were unsuccessful. Errors encountered:\n${failedPolicies.map(e => `Policy [${e.id}] failed to update due to error: ${e.error.message}`).join('\n')}`);
    } else if (updatedPoliciesSuccess.length) {
      log.info(`Done - ${updatedPoliciesSuccess.length} out of ${updatedPoliciesSuccess.length} were successful. No errors encountered.`);
    } else {
      log.error(`Done - all ${failedPolicies.length} failed to update. Errors encountered:\n${failedPolicies.map(e => `Policy [${e.id}] failed to update due to error: ${e.error}`).join('\n')}`);
    }
  }
}
exports.PolicyWatcher = PolicyWatcher;