"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchAgentsData = useFetchAgentsData;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _reactRouterDom = require("react-router-dom");
var _services = require("../../../../../../../common/services");
var _hooks = require("../../../../hooks");
var _services2 = require("../../../../services");
var _constants = require("../../../../constants");
var _get_kuery = require("../utils/get_kuery");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const REFRESH_INTERVAL_MS = 30000;
const MAX_AGENT_ACTIONS = 100;
function useFetchAgentsData() {
  const authz = (0, _hooks.useAuthz)();
  const {
    displayAgentMetrics
  } = _services2.ExperimentalFeaturesService.get();
  const {
    notifications
  } = (0, _hooks.useStartServices)();
  const history = (0, _reactRouterDom.useHistory)();
  const {
    urlParams,
    toUrlParams
  } = (0, _hooks.useUrlParams)();
  const defaultKuery = urlParams.kuery || '';

  // Agent data states
  const [showUpgradeable, setShowUpgradeable] = (0, _react.useState)(false);

  // Table and search states
  const [draftKuery, setDraftKuery] = (0, _react.useState)(defaultKuery);
  const [search, setSearchState] = (0, _react.useState)(defaultKuery);
  const {
    pagination,
    pageSizeOptions,
    setPagination
  } = (0, _hooks.usePagination)();
  const [sortField, setSortField] = (0, _react.useState)('enrolled_at');
  const [sortOrder, setSortOrder] = (0, _react.useState)('desc');
  const VERSION_FIELD = 'local_metadata.elastic.agent.version';
  const HOSTNAME_FIELD = 'local_metadata.host.hostname';

  // Policies state for filtering
  const [selectedAgentPolicies, setSelectedAgentPolicies] = (0, _react.useState)([]);

  // Status for filtering
  const [selectedStatus, setSelectedStatus] = (0, _react.useState)(['healthy', 'unhealthy', 'updating', 'offline']);
  const [selectedTags, setSelectedTags] = (0, _react.useState)([]);
  const showInactive = (0, _react.useMemo)(() => {
    return selectedStatus.some(status => status === 'inactive' || status === 'unenrolled');
  }, [selectedStatus]);
  const setSearch = (0, _react.useCallback)(newVal => {
    setSearchState(newVal);
    if (newVal.trim() === '' && !urlParams.kuery) {
      return;
    }
    if (urlParams.kuery !== newVal) {
      history.replace({
        search: toUrlParams({
          ...urlParams,
          kuery: newVal === '' ? undefined : newVal
        })
      });
    }
  }, [urlParams, history, toUrlParams]);

  // filters kuery
  const kuery = (0, _react.useMemo)(() => {
    return (0, _get_kuery.getKuery)({
      search,
      selectedAgentPolicies,
      selectedTags,
      selectedStatus
    });
  }, [search, selectedAgentPolicies, selectedStatus, selectedTags]);
  const [agentsOnCurrentPage, setAgentsOnCurrentPage] = (0, _react.useState)([]);
  const [agentsStatus, setAgentsStatus] = (0, _react.useState)();
  const [allTags, setAllTags] = (0, _react.useState)();
  const [isLoading, setIsLoading] = (0, _react.useState)(false);
  const [nAgentsInTable, setNAgentsInTable] = (0, _react.useState)(0);
  const [totalInactiveAgents, setTotalInactiveAgents] = (0, _react.useState)(0);
  const [totalManagedAgentIds, setTotalManagedAgentIds] = (0, _react.useState)([]);
  const [managedAgentsOnCurrentPage, setManagedAgentsOnCurrentPage] = (0, _react.useState)(0);
  const [latestAgentActionErrors, setLatestAgentActionErrors] = (0, _react.useState)([]);
  const getSortFieldForAPI = field => {
    if ([VERSION_FIELD, HOSTNAME_FIELD].includes(field)) {
      return `${field}.keyword`;
    }
    return field;
  };
  const isLoadingVar = (0, _react.useRef)(false);

  // Request to fetch agents and agent status
  const currentRequestRef = (0, _react.useRef)(0);
  const fetchData = (0, _react.useCallback)(({
    refreshTags = false
  } = {}) => {
    async function fetchDataAsync() {
      // skipping refresh if previous request is in progress
      if (isLoadingVar.current) {
        return;
      }
      currentRequestRef.current++;
      const currentRequest = currentRequestRef.current;
      isLoadingVar.current = true;
      try {
        var _managedAgentPolicies, _managedAgentPolicies2, _actionStatusResponse;
        setIsLoading(true);
        const [agentsResponse, totalInactiveAgentsResponse, managedAgentPoliciesResponse, agentTagsResponse, actionStatusResponse] = await Promise.all([(0, _hooks.sendGetAgents)({
          page: pagination.currentPage,
          perPage: pagination.pageSize,
          kuery: kuery && kuery !== '' ? kuery : undefined,
          sortField: getSortFieldForAPI(sortField),
          sortOrder,
          showInactive,
          showUpgradeable,
          getStatusSummary: true,
          withMetrics: displayAgentMetrics
        }), (0, _hooks.sendGetAgentStatus)({
          kuery: _services2.AgentStatusKueryHelper.buildKueryForInactiveAgents()
        }), (0, _hooks.sendGetAgentPolicies)({
          kuery: `${_constants.AGENT_POLICY_SAVED_OBJECT_TYPE}.is_managed:true`,
          perPage: _constants.SO_SEARCH_LIMIT,
          full: false
        }), (0, _hooks.sendGetAgentTags)({
          kuery: kuery && kuery !== '' ? kuery : undefined,
          showInactive
        }), (0, _hooks.sendGetActionStatus)({
          latest: REFRESH_INTERVAL_MS + 5000,
          // avoid losing errors
          perPage: MAX_AGENT_ACTIONS
        })]);
        isLoadingVar.current = false;
        // Return if a newer request has been triggered
        if (currentRequestRef.current !== currentRequest) {
          return;
        }
        if (agentsResponse.error) {
          throw agentsResponse.error;
        }
        if (!agentsResponse.data) {
          throw new Error('Invalid GET /agents response');
        }
        if (!totalInactiveAgentsResponse.data) {
          throw new Error('Invalid GET /agents_status response');
        }
        if (managedAgentPoliciesResponse.error) {
          throw new Error(managedAgentPoliciesResponse.error.message);
        }
        if (agentTagsResponse.error) {
          throw agentTagsResponse.error;
        }
        if (!agentTagsResponse.data) {
          throw new Error('Invalid GET /agent/tags response');
        }
        if (actionStatusResponse.error) {
          throw new Error('Invalid GET /agents/action_status response');
        }
        const statusSummary = agentsResponse.data.statusSummary;
        if (!statusSummary) {
          throw new Error('Invalid GET /agents response - no status summary');
        }
        setAgentsStatus((0, _services.agentStatusesToSummary)(statusSummary));
        const newAllTags = agentTagsResponse.data.items;
        // We only want to update the list of available tags if
        // - We haven't set any tags yet
        // - We've received the "refreshTags" flag which will force a refresh of the tags list when an agent is unenrolled
        // - Tags are modified (add, remove, edit)
        if (!allTags || refreshTags || !(0, _lodash.isEqual)(newAllTags, allTags)) {
          setAllTags(newAllTags);
        }
        setAgentsOnCurrentPage(agentsResponse.data.items);
        setNAgentsInTable(agentsResponse.data.total);
        setTotalInactiveAgents(totalInactiveAgentsResponse.data.results.inactive || 0);
        const managedAgentPolicies = (_managedAgentPolicies = (_managedAgentPolicies2 = managedAgentPoliciesResponse.data) === null || _managedAgentPolicies2 === void 0 ? void 0 : _managedAgentPolicies2.items) !== null && _managedAgentPolicies !== void 0 ? _managedAgentPolicies : [];
        if (managedAgentPolicies.length === 0) {
          setTotalManagedAgentIds([]);
          setManagedAgentsOnCurrentPage(0);
        } else {
          var _response$data$items, _response$data;
          // Find all the agents that have managed policies and are not unenrolled
          const policiesKuery = managedAgentPolicies.map(policy => `policy_id:"${policy.id}"`).join(' or ');
          const response = await (0, _hooks.sendGetAgents)({
            kuery: `NOT (status:unenrolled) and ${policiesKuery}`,
            perPage: _constants.SO_SEARCH_LIMIT,
            showInactive: true
          });
          if (response.error) {
            throw new Error(response.error.message);
          }
          const allManagedAgents = (_response$data$items = (_response$data = response.data) === null || _response$data === void 0 ? void 0 : _response$data.items) !== null && _response$data$items !== void 0 ? _response$data$items : [];
          const allManagedAgentIds = allManagedAgents === null || allManagedAgents === void 0 ? void 0 : allManagedAgents.map(agent => agent.id);
          setTotalManagedAgentIds(allManagedAgentIds);
          setManagedAgentsOnCurrentPage(agentsResponse.data.items.map(agent => agent.id).filter(agentId => allManagedAgentIds.includes(agentId)).length);
        }
        const actionErrors = ((_actionStatusResponse = actionStatusResponse.data) === null || _actionStatusResponse === void 0 ? void 0 : _actionStatusResponse.items.filter(action => {
          var _action$latestErrors$, _action$latestErrors;
          return (_action$latestErrors$ = (_action$latestErrors = action.latestErrors) === null || _action$latestErrors === void 0 ? void 0 : _action$latestErrors.length) !== null && _action$latestErrors$ !== void 0 ? _action$latestErrors$ : 0 > 1;
        }).map(action => action.actionId)) || [];
        const allRecentActionErrors = [...new Set([...latestAgentActionErrors, ...actionErrors])];
        if (!(0, _lodash.isEqual)(latestAgentActionErrors, allRecentActionErrors)) {
          setLatestAgentActionErrors(allRecentActionErrors);
        }
      } catch (error) {
        notifications.toasts.addError(error, {
          title: _i18n.i18n.translate('xpack.fleet.agentList.errorFetchingDataTitle', {
            defaultMessage: 'Error fetching agents'
          })
        });
      }
      setIsLoading(false);
    }
    fetchDataAsync();
  }, [pagination.currentPage, pagination.pageSize, kuery, sortField, sortOrder, showInactive, showUpgradeable, displayAgentMetrics, allTags, latestAgentActionErrors, notifications.toasts]);

  // Send request to get agent list and status
  (0, _react.useEffect)(() => {
    fetchData();
    const interval = setInterval(() => {
      fetchData();
    }, REFRESH_INTERVAL_MS);
    return () => clearInterval(interval);
  }, [fetchData]);
  const agentPoliciesRequest = (0, _hooks.useGetAgentPolicies)({
    page: 1,
    perPage: _constants.SO_SEARCH_LIMIT,
    full: authz.fleet.readAgentPolicies
  });
  const agentPolicies = (0, _react.useMemo)(() => agentPoliciesRequest.data ? agentPoliciesRequest.data.items : [], [agentPoliciesRequest]);
  const agentPoliciesIndexedById = (0, _react.useMemo)(() => {
    return agentPolicies.reduce((acc, agentPolicy) => {
      acc[agentPolicy.id] = agentPolicy;
      return acc;
    }, {});
  }, [agentPolicies]);
  return {
    allTags,
    setAllTags,
    agentsOnCurrentPage,
    agentsStatus,
    isLoading,
    nAgentsInTable,
    totalInactiveAgents,
    totalManagedAgentIds,
    managedAgentsOnCurrentPage,
    showUpgradeable,
    setShowUpgradeable,
    search,
    setSearch,
    selectedAgentPolicies,
    setSelectedAgentPolicies,
    sortField,
    setSortField,
    sortOrder,
    setSortOrder,
    selectedStatus,
    setSelectedStatus,
    selectedTags,
    setSelectedTags,
    agentPolicies,
    agentPoliciesRequest,
    agentPoliciesIndexedById,
    pagination,
    pageSizeOptions,
    setPagination,
    kuery,
    draftKuery,
    setDraftKuery,
    fetchData,
    currentRequestRef,
    latestAgentActionErrors,
    setLatestAgentActionErrors
  };
}