"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAstContext = getAstContext;
exports.removeMarkerArgFromArgsList = removeMarkerArgFromArgsList;
var _settings = require("../definitions/settings");
var _constants = require("./constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function findNode(nodes, offset) {
  for (const node of nodes) {
    if (Array.isArray(node)) {
      const ret = findNode(node, offset);
      if (ret) {
        return ret;
      }
    } else {
      if (node.location.min <= offset && node.location.max >= offset) {
        if ('args' in node) {
          const ret = findNode(node.args, offset);
          // if the found node is the marker, then return its parent
          if ((ret === null || ret === void 0 ? void 0 : ret.text) === _constants.EDITOR_MARKER) {
            return node;
          }
          if (ret) {
            return ret;
          }
        }
        return node;
      }
    }
  }
}
function findCommand(ast, offset) {
  const commandIndex = ast.findIndex(({
    location
  }) => location.min <= offset && location.max >= offset);
  return ast[commandIndex] || ast[ast.length - 1];
}
function findOption(nodes, offset) {
  return findCommandSubType(nodes, offset, _helpers.isOptionItem);
}
function findSetting(nodes, offset) {
  return findCommandSubType(nodes, offset, _helpers.isSettingItem);
}
function findCommandSubType(nodes, offset, isOfTypeFn) {
  for (const node of nodes) {
    if (isOfTypeFn(node)) {
      if (node.location.min <= offset && node.location.max >= offset || nodes[nodes.length - 1] === node && node.location.max < offset) {
        return node;
      }
    }
  }
}
function isMarkerNode(node) {
  return Boolean(node && ((0, _helpers.isColumnItem)(node) || (0, _helpers.isSourceItem)(node)) && node.name.endsWith(_constants.EDITOR_MARKER));
}
function cleanMarkerNode(node) {
  return isMarkerNode(node) ? undefined : node;
}
function isNotMarkerNodeOrArray(arg) {
  return Array.isArray(arg) || !isMarkerNode(arg);
}
function mapToNonMarkerNode(arg) {
  return Array.isArray(arg) ? arg.filter(isNotMarkerNodeOrArray).map(mapToNonMarkerNode) : arg;
}
function removeMarkerArgFromArgsList(node) {
  if (!node) {
    return;
  }
  if (node.type === 'command' || node.type === 'option' || node.type === 'function') {
    return {
      ...node,
      args: node.args.filter(isNotMarkerNodeOrArray).map(mapToNonMarkerNode)
    };
  }
  return node;
}
function findAstPosition(ast, offset) {
  const command = findCommand(ast, offset);
  if (!command) {
    return {
      command: undefined,
      node: undefined,
      option: undefined,
      setting: undefined
    };
  }
  return {
    command: removeMarkerArgFromArgsList(command),
    option: removeMarkerArgFromArgsList(findOption(command.args, offset)),
    node: removeMarkerArgFromArgsList(cleanMarkerNode(findNode(command.args, offset))),
    setting: removeMarkerArgFromArgsList(findSetting(command.args, offset))
  };
}
function isNotEnrichClauseAssigment(node, command) {
  return node.name !== '=' && command.name !== 'enrich';
}
function isBuiltinFunction(node) {
  var _getFunctionDefinitio;
  return ((_getFunctionDefinitio = (0, _helpers.getFunctionDefinition)(node.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === 'builtin';
}

/**
 * Given a ES|QL query string, its AST and the cursor position,
 * it returns the type of context for the position ("list", "function", "option", "setting", "expression", "newCommand")
 * plus the whole hierarchy of nodes (command, option, setting and actual position node) context.
 *
 * Type details:
 * * "list": the cursor is inside a "in" list of values (i.e. `a in (1, 2, <here>)`)
 * * "function": the cursor is inside a function call (i.e. `fn(<here>)`)
 * * "option": the cursor is inside a command option (i.e. `command ... by <here>`)
 * * "setting": the cursor is inside a setting (i.e. `command _<here>`)
 * * "expression": the cursor is inside a command expression (i.e. `command ... <here>` or `command a = ... <here>`)
 * * "newCommand": the cursor is at the beginning of a new command (i.e. `command1 | command2 | <here>`)
 */
function getAstContext(queryString, ast, offset) {
  const {
    command,
    option,
    setting,
    node
  } = findAstPosition(ast, offset);
  if (node) {
    if (node.type === 'function') {
      if (['in', 'not_in'].includes(node.name) && Array.isArray(node.args[1])) {
        // command ... a in ( <here> )
        return {
          type: 'list',
          command,
          node,
          option,
          setting
        };
      }
      if (isNotEnrichClauseAssigment(node, command) && !isBuiltinFunction(node)) {
        // command ... fn( <here> )
        return {
          type: 'function',
          command,
          node,
          option,
          setting
        };
      }
    }
    if (node.type === 'option' || option) {
      // command ... by <here>
      return {
        type: 'option',
        command,
        node,
        option,
        setting
      };
    }
    // for now it's only an enrich thing
    if (node.type === 'source' && node.text === _settings.ENRICH_MODES.prefix) {
      // command _<here>
      return {
        type: 'setting',
        command,
        node,
        option,
        setting
      };
    }
  }
  if (!command || queryString.length <= offset && (0, _helpers.getLastCharFromTrimmed)(queryString) === '|') {
    //   // ... | <here>
    return {
      type: 'newCommand',
      command: undefined,
      node,
      option,
      setting
    };
  }
  if (command && command.args.length) {
    if (option) {
      return {
        type: 'option',
        command,
        node,
        option,
        setting
      };
    }
  }

  // command a ... <here> OR command a = ... <here>
  return {
    type: 'expression',
    command,
    option,
    node,
    setting
  };
}