"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.builtinFunctions = void 0;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function createMathDefinition(name, types, description, validate) {
  return {
    type: 'builtin',
    name,
    description,
    supportedCommands: ['eval', 'where', 'row', 'stats', 'sort'],
    supportedOptions: ['by'],
    signatures: types.map(type => {
      if (Array.isArray(type)) {
        return {
          params: [{
            name: 'left',
            type: type[0]
          }, {
            name: 'right',
            type: type[1]
          }],
          returnType: /literal/.test(type[0]) ? type[1] : type[0]
        };
      }
      return {
        params: [{
          name: 'left',
          type
        }, {
          name: 'right',
          type
        }],
        returnType: type
      };
    }),
    validate
  };
}
function createComparisonDefinition({
  name,
  description,
  extraSignatures = []
}, validate) {
  return {
    type: 'builtin',
    name,
    description,
    supportedCommands: ['eval', 'where', 'row', 'sort'],
    supportedOptions: ['by'],
    validate,
    signatures: [{
      params: [{
        name: 'left',
        type: 'number'
      }, {
        name: 'right',
        type: 'number'
      }],
      returnType: 'boolean'
    }, {
      params: [{
        name: 'left',
        type: 'string'
      }, {
        name: 'right',
        type: 'string'
      }],
      returnType: 'boolean'
    }, {
      params: [{
        name: 'left',
        type: 'date'
      }, {
        name: 'right',
        type: 'date'
      }],
      returnType: 'boolean'
    }, {
      params: [{
        name: 'left',
        type: 'ip'
      }, {
        name: 'right',
        type: 'ip'
      }],
      returnType: 'boolean'
    }, {
      params: [{
        name: 'left',
        type: 'version'
      }, {
        name: 'right',
        type: 'version'
      }],
      returnType: 'boolean'
    },
    // constant strings okay because of implicit casting for
    // string to version and ip
    //
    // boolean casting is handled on the specific comparison function
    // that support booleans
    //
    // date casting is handled in the validation routine since it's a
    // general rule. Look in compareLiteralType()
    ...['ip', 'version'].flatMap(type => [{
      params: [{
        name: 'left',
        type
      }, {
        name: 'right',
        type: 'string',
        constantOnly: true
      }],
      returnType: 'boolean'
    }, {
      params: [{
        name: 'right',
        type: 'string',
        constantOnly: true
      }, {
        name: 'right',
        type
      }],
      returnType: 'boolean'
    }]), ...extraSignatures]
  };
}
const builtinFunctions = exports.builtinFunctions = [createMathDefinition('+', ['number', ['date', 'time_literal'], ['time_literal', 'date']], _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.addDoc', {
  defaultMessage: 'Add (+)'
})), createMathDefinition('-', ['number', ['date', 'time_literal'], ['time_literal', 'date']], _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.subtractDoc', {
  defaultMessage: 'Subtract (-)'
})), createMathDefinition('*', ['number'], _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.multiplyDoc', {
  defaultMessage: 'Multiply (*)'
})), createMathDefinition('/', ['number'], _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.divideDoc', {
  defaultMessage: 'Divide (/)'
}), fnDef => {
  const [left, right] = fnDef.args;
  const messages = [];
  if (!Array.isArray(left) && !Array.isArray(right)) {
    if (right.type === 'literal' && right.literalType === 'number') {
      if (right.value === 0) {
        messages.push({
          type: 'warning',
          code: 'divideByZero',
          text: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.divide.warning.divideByZero', {
            defaultMessage: 'Cannot divide by zero: {left}/{right}',
            values: {
              left: left.text,
              right: right.value
            }
          }),
          location: fnDef.location
        });
      }
    }
  }
  return messages;
}), createMathDefinition('%', ['number'], _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.moduleDoc', {
  defaultMessage: 'Module (%)'
}), fnDef => {
  const [left, right] = fnDef.args;
  const messages = [];
  if (!Array.isArray(left) && !Array.isArray(right)) {
    if (right.type === 'literal' && right.literalType === 'number') {
      if (right.value === 0) {
        messages.push({
          type: 'warning',
          code: 'moduleByZero',
          text: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.divide.warning.zeroModule', {
            defaultMessage: 'Module by zero can return null value: {left}%{right}',
            values: {
              left: left.text,
              right: right.value
            }
          }),
          location: fnDef.location
        });
      }
    }
  }
  return messages;
}), ...[{
  name: '==',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.equalToDoc', {
    defaultMessage: 'Equal to'
  }),
  extraSignatures: [{
    params: [{
      name: 'left',
      type: 'boolean'
    }, {
      name: 'right',
      type: 'boolean'
    }],
    returnType: 'boolean'
  },
  // constant strings okay because of implicit casting
  {
    params: [{
      name: 'left',
      type: 'boolean'
    }, {
      name: 'right',
      type: 'string',
      constantOnly: true
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'right',
      type: 'string',
      constantOnly: true
    }, {
      name: 'right',
      type: 'boolean'
    }],
    returnType: 'boolean'
  }]
}, {
  name: '!=',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.notEqualToDoc', {
    defaultMessage: 'Not equal to'
  }),
  extraSignatures: [{
    params: [{
      name: 'left',
      type: 'boolean'
    }, {
      name: 'right',
      type: 'boolean'
    }],
    returnType: 'boolean'
  },
  // constant strings okay because of implicit casting
  {
    params: [{
      name: 'left',
      type: 'boolean'
    }, {
      name: 'right',
      type: 'string',
      constantOnly: true
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'right',
      type: 'string',
      constantOnly: true
    }, {
      name: 'right',
      type: 'boolean'
    }],
    returnType: 'boolean'
  }]
}, {
  name: '<',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.lessThanDoc', {
    defaultMessage: 'Less than'
  })
}, {
  name: '>',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.greaterThanDoc', {
    defaultMessage: 'Greater than'
  })
}, {
  name: '<=',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.lessThanOrEqualToDoc', {
    defaultMessage: 'Less than or equal to'
  })
}, {
  name: '>=',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.greaterThanOrEqualToDoc', {
    defaultMessage: 'Greater than or equal to'
  })
}].map(op => createComparisonDefinition(op)), ...[
// Skip the insensitive case equality until it gets restored back
// new special comparison operator for strings only
// {
//   name: '=~',
//   description: i18n.translate('kbn-esql-validation-autocomplete.esql.definition.equalToCaseInsensitiveDoc', {
//     defaultMessage: 'Case insensitive equality',
//   }),
// },
{
  name: 'like',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.likeDoc', {
    defaultMessage: 'Filter data based on string patterns'
  })
}, {
  name: 'not_like',
  description: ''
}, {
  name: 'rlike',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.rlikeDoc', {
    defaultMessage: 'Filter data based on string regular expressions'
  })
}, {
  name: 'not_rlike',
  description: ''
}].map(({
  name,
  description
}) => ({
  type: 'builtin',
  ignoreAsSuggestion: /not/.test(name),
  name,
  description,
  supportedCommands: ['eval', 'where', 'row', 'sort'],
  supportedOptions: ['by'],
  signatures: [{
    params: [{
      name: 'left',
      type: 'string'
    }, {
      name: 'right',
      type: 'string'
    }],
    returnType: 'boolean'
  }]
})), ...[{
  name: 'in',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.inDoc', {
    defaultMessage: 'Tests if the value an expression takes is contained in a list of other expressions'
  })
}, {
  name: 'not_in',
  description: ''
}].map(({
  name,
  description
}) => ({
  // set all arrays to type "any" for now
  // this only applies to the "in" operator
  // e.g. "foo" in ( "foo", "bar" )
  //
  // we did this because the "in" operator now supports
  // mixed-type arrays like ( "1.2.3", versionVar )
  // because of implicit casting.
  //
  // we need to revisit with more robust validation
  type: 'builtin',
  ignoreAsSuggestion: /not/.test(name),
  name,
  description,
  supportedCommands: ['eval', 'where', 'row', 'sort'],
  signatures: [{
    params: [{
      name: 'left',
      type: 'number'
    }, {
      name: 'right',
      type: 'any[]'
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'left',
      type: 'string'
    }, {
      name: 'right',
      type: 'any[]'
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'left',
      type: 'boolean'
    }, {
      name: 'right',
      type: 'any[]'
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'left',
      type: 'date'
    }, {
      name: 'right',
      type: 'any[]'
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'left',
      type: 'version'
    }, {
      name: 'right',
      type: 'any[]'
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'left',
      type: 'ip'
    }, {
      name: 'right',
      type: 'any[]'
    }],
    returnType: 'boolean'
  }]
})), ...[{
  name: 'and',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.andDoc', {
    defaultMessage: 'and'
  })
}, {
  name: 'or',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.orDoc', {
    defaultMessage: 'or'
  })
}].map(({
  name,
  description
}) => ({
  type: 'builtin',
  name,
  description,
  supportedCommands: ['eval', 'where', 'row', 'sort'],
  supportedOptions: ['by'],
  signatures: [{
    params: [{
      name: 'left',
      type: 'boolean'
    }, {
      name: 'right',
      type: 'boolean'
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'left',
      type: 'null'
    }, {
      name: 'right',
      type: 'boolean'
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'left',
      type: 'boolean'
    }, {
      name: 'right',
      type: 'null'
    }],
    returnType: 'boolean'
  }, {
    params: [{
      name: 'left',
      type: 'null'
    }, {
      name: 'right',
      type: 'null'
    }],
    returnType: 'boolean'
  }]
})), {
  type: 'builtin',
  name: 'not',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.notDoc', {
    defaultMessage: 'Not'
  }),
  supportedCommands: ['eval', 'where', 'row', 'sort'],
  supportedOptions: ['by'],
  signatures: [{
    params: [{
      name: 'expression',
      type: 'boolean'
    }],
    returnType: 'boolean'
  }]
}, ...[{
  name: 'is null',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.isNullDoc', {
    defaultMessage: 'Predicate for NULL comparison: returns true if the value is NULL'
  })
}, {
  name: 'is not null',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.isNotNullDoc', {
    defaultMessage: 'Predicate for NULL comparison: returns true if the value is not NULL'
  })
}].map(({
  name,
  description
}) => ({
  type: 'builtin',
  name,
  description,
  supportedCommands: ['eval', 'where', 'row', 'sort'],
  signatures: [{
    params: [{
      name: 'left',
      type: 'any'
    }],
    returnType: 'boolean'
  }]
})), {
  type: 'builtin',
  name: '=',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.assignDoc', {
    defaultMessage: 'Assign (=)'
  }),
  supportedCommands: ['eval', 'stats', 'row', 'dissect', 'where', 'enrich'],
  supportedOptions: ['by', 'with'],
  signatures: [{
    params: [{
      name: 'left',
      type: 'any'
    }, {
      name: 'right',
      type: 'any'
    }],
    returnType: 'void'
  }]
}, {
  name: 'functions',
  type: 'builtin',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.functionsDoc', {
    defaultMessage: 'Show ES|QL avaialble functions with signatures'
  }),
  supportedCommands: ['meta'],
  signatures: [{
    params: [],
    returnType: 'void'
  }]
}, {
  name: 'info',
  type: 'builtin',
  description: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definition.infoDoc', {
    defaultMessage: 'Show information about the current ES node'
  }),
  supportedCommands: ['show'],
  signatures: [{
    params: [],
    returnType: 'void'
  }]
}];