"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.computeLocationExtends = computeLocationExtends;
exports.createColumn = createColumn;
exports.createColumnStar = createColumnStar;
exports.createCommand = createCommand;
exports.createFakeMultiplyLiteral = createFakeMultiplyLiteral;
exports.createFunction = createFunction;
exports.createList = createList;
exports.createLiteral = createLiteral;
exports.createLiteralString = createLiteralString;
exports.createNumericLiteral = createNumericLiteral;
exports.createOption = createOption;
exports.createPolicy = createPolicy;
exports.createSetting = createSetting;
exports.createSource = createSource;
exports.createTimeUnit = createTimeUnit;
exports.nonNullable = nonNullable;
exports.sanitizeIdentifierString = sanitizeIdentifierString;
exports.textExistsAndIsValid = textExistsAndIsValid;
exports.wrapIdentifierAsArray = wrapIdentifierAsArray;
var _ast_position_utils = require("./ast_position_utils");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * In case of changes in the grammar, this script should be updated: esql_update_ast_script.js
 */

function nonNullable(v) {
  return v != null;
}
function createCommand(name, ctx) {
  return {
    type: 'command',
    name,
    text: ctx.getText(),
    args: [],
    location: (0, _ast_position_utils.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
function createList(ctx, values) {
  return {
    type: 'list',
    name: ctx.getText(),
    values,
    text: ctx.getText(),
    location: (0, _ast_position_utils.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
function createNumericLiteral(ctx) {
  const text = ctx.getText();
  return {
    type: 'literal',
    literalType: 'number',
    text,
    name: text,
    value: Number(text),
    location: (0, _ast_position_utils.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
function createFakeMultiplyLiteral(ctx) {
  return {
    type: 'literal',
    literalType: 'number',
    text: ctx.getText(),
    name: ctx.getText(),
    value: ctx.PLUS() ? 1 : -1,
    location: (0, _ast_position_utils.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception)
  };
}
function createLiteralString(token) {
  const text = token.text;
  return {
    type: 'literal',
    literalType: 'string',
    text,
    name: text,
    value: text,
    location: (0, _ast_position_utils.getPosition)(token),
    incomplete: Boolean(token.text === '')
  };
}
function isMissingText(text) {
  return /<missing /.test(text);
}
function textExistsAndIsValid(text) {
  return !!(text && !isMissingText(text));
}
function createLiteral(type, node) {
  if (!node) {
    return;
  }
  const text = node.getText();
  const partialLiteral = {
    type: 'literal',
    text,
    name: text,
    location: (0, _ast_position_utils.getPosition)(node.symbol),
    incomplete: isMissingText(text)
  };
  if (type === 'number') {
    return {
      ...partialLiteral,
      literalType: type,
      value: Number(text)
    };
  }
  return {
    ...partialLiteral,
    literalType: type,
    value: text
  };
}
function createTimeUnit(ctx) {
  return {
    type: 'timeInterval',
    quantity: Number(ctx.integerValue().INTEGER_LITERAL().getText()),
    unit: ctx.UNQUOTED_IDENTIFIER().symbol.text,
    text: ctx.getText(),
    location: (0, _ast_position_utils.getPosition)(ctx.start, ctx.stop),
    name: `${ctx.integerValue().INTEGER_LITERAL().getText()} ${ctx.UNQUOTED_IDENTIFIER().symbol.text}`,
    incomplete: Boolean(ctx.exception)
  };
}
function createFunction(name, ctx, customPosition) {
  return {
    type: 'function',
    name,
    text: ctx.getText(),
    location: customPosition !== null && customPosition !== void 0 ? customPosition : (0, _ast_position_utils.getPosition)(ctx.start, ctx.stop),
    args: [],
    incomplete: Boolean(ctx.exception)
  };
}
function walkFunctionStructure(args, initialLocation, prop, getNextItemIndex) {
  let nextArg = args[getNextItemIndex(args)];
  const location = {
    ...initialLocation
  };
  while (Array.isArray(nextArg) || nextArg) {
    if (Array.isArray(nextArg)) {
      nextArg = nextArg[getNextItemIndex(nextArg)];
    } else {
      location[prop] = Math[prop](location[prop], nextArg.location[prop]);
      if (nextArg.type === 'function') {
        nextArg = nextArg.args[getNextItemIndex(nextArg.args)];
      } else {
        nextArg = undefined;
      }
    }
  }
  return location[prop];
}
function computeLocationExtends(fn) {
  const location = fn.location;
  if (fn.args) {
    // get min location navigating in depth keeping the left/first arg
    location.min = walkFunctionStructure(fn.args, location, 'min', () => 0);
    // get max location navigating in depth keeping the right/last arg
    location.max = walkFunctionStructure(fn.args, location, 'max', args => args.length - 1);
    // in case of empty array as last arg, bump the max location by 3 chars (empty brackets)
    if (Array.isArray(fn.args[fn.args.length - 1]) && !fn.args[fn.args.length - 1].length) {
      location.max += 3;
    }
  }
  return location;
}

// Note: do not import esql_parser or bundle size will grow up by ~500 kb
/**
 * Do not touch this piece of code as it is auto-generated by a script
 */

/* SCRIPT_MARKER_START */
function getQuotedText(ctx) {
  return [27 /* esql_parser.QUOTED_STRING */, 67 /* esql_parser.QUOTED_IDENTIFIER */].map(keyCode => ctx.getToken(keyCode, 0)).filter(nonNullable)[0];
}
function getUnquotedText(ctx) {
  return [66 /* esql_parser.UNQUOTED_IDENTIFIER */, 72 /* esql_parser.FROM_UNQUOTED_IDENTIFIER */].map(keyCode => ctx.getToken(keyCode, 0)).filter(nonNullable)[0];
}
/* SCRIPT_MARKER_END */

function isQuoted(text) {
  return text && /^(`)/.test(text);
}

/**
 * Follow a similar logic to the ES one:
 * * remove backticks at the beginning and at the end
 * * remove double backticks
 */
function safeBackticksRemoval(text) {
  return (text === null || text === void 0 ? void 0 : text.replace(_constants.TICKS_REGEX, '').replace(_constants.DOUBLE_TICKS_REGEX, _constants.SINGLE_BACKTICK)) || '';
}
function sanitizeIdentifierString(ctx) {
  var _getUnquotedText, _getQuotedText;
  const result = ((_getUnquotedText = getUnquotedText(ctx)) === null || _getUnquotedText === void 0 ? void 0 : _getUnquotedText.getText()) || safeBackticksRemoval((_getQuotedText = getQuotedText(ctx)) === null || _getQuotedText === void 0 ? void 0 : _getQuotedText.getText()) || safeBackticksRemoval(ctx.getText()); // for some reason some quoted text is not detected correctly by the parser

  // TODO - understand why <missing null> is now returned as the match text for the FROM command
  return result === '<missing null>' ? '' : result;
}
function wrapIdentifierAsArray(identifierCtx) {
  return Array.isArray(identifierCtx) ? identifierCtx : [identifierCtx];
}
function createSetting(policyName, mode) {
  return {
    type: 'mode',
    name: mode.replace('_', '').toLowerCase(),
    text: mode,
    location: (0, _ast_position_utils.getPosition)(policyName, {
      stop: policyName.start + mode.length - 1
    }),
    // unfortunately this is the only location we have
    incomplete: false
  };
}

/**
 * In https://github.com/elastic/elasticsearch/pull/103949 the ENRICH policy name
 * changed from rule to token type so we need to handle this specifically
 */
function createPolicy(token, policy) {
  return {
    type: 'source',
    name: policy,
    text: policy,
    sourceType: 'policy',
    location: (0, _ast_position_utils.getPosition)({
      start: token.stop - policy.length + 1,
      stop: token.stop
    }),
    // take into account ccq modes
    incomplete: false
  };
}
function createSource(ctx, type = 'index') {
  const text = sanitizeIdentifierString(ctx);
  return {
    type: 'source',
    name: text,
    sourceType: type,
    text,
    location: (0, _ast_position_utils.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception || text === '')
  };
}
function createColumnStar(ctx) {
  return {
    type: 'column',
    name: ctx.getText(),
    text: ctx.getText(),
    location: (0, _ast_position_utils.getPosition)(ctx.symbol),
    incomplete: ctx.getText() === '',
    quoted: false
  };
}
function createColumn(ctx) {
  const text = sanitizeIdentifierString(ctx);
  const hasQuotes = Boolean(getQuotedText(ctx) || isQuoted(ctx.getText()));
  return {
    type: 'column',
    name: text,
    text: ctx.getText(),
    location: (0, _ast_position_utils.getPosition)(ctx.start, ctx.stop),
    incomplete: Boolean(ctx.exception || text === ''),
    quoted: hasQuotes
  };
}
function createOption(name, ctx) {
  var _ctx$children;
  return {
    type: 'option',
    name,
    text: ctx.getText(),
    location: (0, _ast_position_utils.getPosition)(ctx.start, ctx.stop),
    args: [],
    incomplete: Boolean(ctx.exception || ((_ctx$children = ctx.children) === null || _ctx$children === void 0 ? void 0 : _ctx$children.some(c => {
      // @ts-expect-error not exposed in type but exists see https://github.com/antlr/antlr4/blob/v4.11.1/runtime/JavaScript/src/antlr4/tree/ErrorNodeImpl.js#L19
      return Boolean(c.isErrorNode);
    })))
  };
}