"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KQL_FUNCTION_IS = void 0;
exports.buildNodeParams = buildNodeParams;
exports.isNode = isNode;
exports.toElasticsearchQuery = toElasticsearchQuery;
exports.toKqlExpression = toKqlExpression;
var _lodash = require("lodash");
var _filters = require("../../filters");
var _get_fields = require("./utils/get_fields");
var _utils = require("../../utils");
var _get_full_field_name_node = require("./utils/get_full_field_name_node");
var ast = _interopRequireWildcard(require("../ast"));
var literal = _interopRequireWildcard(require("../node_types/literal"));
var wildcard = _interopRequireWildcard(require("../node_types/wildcard"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const KQL_FUNCTION_IS = exports.KQL_FUNCTION_IS = 'is';
function isNode(node) {
  return node.function === KQL_FUNCTION_IS;
}
function buildNodeParams(fieldName, value) {
  if ((0, _lodash.isUndefined)(fieldName)) {
    throw new Error('fieldName is a required argument');
  }
  if ((0, _lodash.isUndefined)(value)) {
    throw new Error('value is a required argument');
  }
  const fieldNode = typeof fieldName === 'string' ? ast.fromLiteralExpression(fieldName) : literal.buildNode(fieldName);
  const valueNode = typeof value === 'string' ? ast.fromLiteralExpression(value) : literal.buildNode(value);
  return {
    arguments: [fieldNode, valueNode]
  };
}
function toElasticsearchQuery(node, indexPattern, config = {}, context = {}) {
  const {
    arguments: [fieldNameArg, valueArg]
  } = node;
  const isExistsQuery = wildcard.isNode(valueArg) && wildcard.isLoneWildcard(valueArg);
  const isAllFieldsQuery = wildcard.isNode(fieldNameArg) && wildcard.isLoneWildcard(fieldNameArg);
  const isMatchAllQuery = isExistsQuery && isAllFieldsQuery;
  if (isMatchAllQuery) {
    return {
      match_all: {}
    };
  }
  const fullFieldNameArg = (0, _get_full_field_name_node.getFullFieldNameNode)(fieldNameArg, indexPattern, context !== null && context !== void 0 && context.nested ? context.nested.path : undefined);
  const value = !(0, _lodash.isUndefined)(valueArg) ? ast.toElasticsearchQuery(valueArg) : valueArg;
  const type = valueArg.isQuoted ? 'phrase' : 'best_fields';
  if (fullFieldNameArg.value === null) {
    if (wildcard.isNode(valueArg)) {
      return {
        query_string: {
          query: wildcard.toQueryStringQuery(valueArg)
        }
      };
    }
    return {
      multi_match: {
        type,
        query: value,
        lenient: true
      }
    };
  }
  const fields = indexPattern ? (0, _get_fields.getFields)(fullFieldNameArg, indexPattern) : [];
  // If no fields are found in the index pattern we send through the given field name as-is. We do this to preserve
  // the behaviour of lucene on dashboards where there are panels based on different index patterns that have different
  // fields. If a user queries on a field that exists in one pattern but not the other, the index pattern without the
  // field should return no results. It's debatable whether this is desirable, but it's been that way forever, so we'll
  // keep things familiar for now.
  if (fields && fields.length === 0) {
    fields.push({
      name: ast.toElasticsearchQuery(fullFieldNameArg),
      scripted: false,
      type: ''
    });
  }

  // Special case for wildcards where there are no fields or all fields share the same prefix
  if (isExistsQuery && (!(fields !== null && fields !== void 0 && fields.length) || (fields === null || fields === void 0 ? void 0 : fields.length) === (indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.fields.length))) {
    return {
      match_all: {}
    };
  }
  const queries = fields.reduce((accumulator, field) => {
    var _field$esTypes;
    const isKeywordField = ((_field$esTypes = field.esTypes) === null || _field$esTypes === void 0 ? void 0 : _field$esTypes.length) === 1 && field.esTypes.includes('keyword');
    const wrapWithNestedQuery = query => {
      // Wildcards can easily include nested and non-nested fields. There isn't a good way to let
      // users handle this themselves so we automatically add nested queries in this scenario.
      const subTypeNested = (0, _utils.getDataViewFieldSubtypeNested)(field);
      if (!wildcard.isNode(fullFieldNameArg) || !(subTypeNested !== null && subTypeNested !== void 0 && subTypeNested.nested) || context !== null && context !== void 0 && context.nested) {
        return query;
      } else {
        return {
          nested: {
            path: subTypeNested.nested.path,
            query,
            score_mode: 'none',
            ...(typeof config.nestedIgnoreUnmapped === 'boolean' && {
              ignore_unmapped: config.nestedIgnoreUnmapped
            })
          }
        };
      }
    };
    if (field.scripted) {
      // Exists queries don't make sense for scripted fields
      if (!isExistsQuery) {
        return [...accumulator, {
          script: {
            ...(0, _filters.getPhraseScript)(field, value)
          }
        }];
      }
    } else if (isExistsQuery) {
      return [...accumulator, wrapWithNestedQuery({
        exists: {
          field: field.name
        }
      })];
    } else if (wildcard.isNode(valueArg)) {
      const query = isKeywordField ? {
        wildcard: {
          [field.name]: {
            value: wildcard.toQueryStringQuery(valueArg),
            ...(typeof config.caseInsensitive === 'boolean' && {
              case_insensitive: config.caseInsensitive
            })
          }
        }
      } : {
        query_string: {
          fields: [field.name],
          query: wildcard.toQueryStringQuery(valueArg)
        }
      };
      return [...accumulator, wrapWithNestedQuery(query)];
    } else if (field.type === 'date') {
      /*
      If we detect that it's a date field and the user wants an exact date, we need to convert the query to both >= and <= the value provided to force a range query. This is because match and match_phrase queries do not accept a timezone parameter.
      dateFormatTZ can have the value of 'Browser', in which case we guess the timezone using moment.tz.guess.
      */
      const timeZoneParam = config.dateFormatTZ ? {
        time_zone: (0, _utils.getTimeZoneFromSettings)(config.dateFormatTZ)
      } : {};
      return [...accumulator, wrapWithNestedQuery({
        range: {
          [field.name]: {
            gte: value,
            lte: value,
            ...timeZoneParam
          }
        }
      })];
    } else if (isKeywordField) {
      return [...accumulator, wrapWithNestedQuery({
        term: {
          [field.name]: {
            value,
            ...(typeof config.caseInsensitive === 'boolean' && {
              case_insensitive: config.caseInsensitive
            })
          }
        }
      })];
    } else {
      const queryType = type === 'phrase' ? 'match_phrase' : 'match';
      return [...accumulator, wrapWithNestedQuery({
        [queryType]: {
          [field.name]: value
        }
      })];
    }
  }, []);
  return {
    bool: {
      should: queries || [],
      minimum_should_match: 1
    }
  };
}
function toKqlExpression(node) {
  const [field, value] = node.arguments;
  if (field.value === null) return `${ast.toKqlExpression(value)}`;
  return `${ast.toKqlExpression(field)}: ${ast.toKqlExpression(value)}`;
}