"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toElasticsearchQuery = exports.fromLiteralExpression = exports.fromKueryExpression = void 0;
exports.toKqlExpression = toKqlExpression;
var _node_types = require("../node_types");
var _kuery_syntax_error = require("../kuery_syntax_error");
var _grammar = require("../grammar");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const fromExpression = (expression, parseOptions = {}, parse = _grammar.parse) => {
  if (typeof expression === 'undefined') {
    throw new Error('expression must be a string, got undefined instead');
  }
  return parse(expression, {
    ...parseOptions,
    helpers: {
      nodeTypes: _node_types.nodeTypes
    }
  });
};
const fromLiteralExpression = (expression, parseOptions = {}) => {
  return fromExpression(expression, {
    ...parseOptions,
    startRule: 'Literal'
  }, _grammar.parse);
};
exports.fromLiteralExpression = fromLiteralExpression;
const fromKueryExpression = (expression, parseOptions = {}) => {
  try {
    return fromExpression(expression, parseOptions, _grammar.parse);
  } catch (error) {
    if (error.name === 'SyntaxError') {
      throw new _kuery_syntax_error.KQLSyntaxError(error, expression);
    } else {
      throw error;
    }
  }
};

/**
 * Given a KQL AST node, generate the corresponding KQL expression.
 * @public
 * @param node
 */
exports.fromKueryExpression = fromKueryExpression;
function toKqlExpression(node) {
  if (_node_types.nodeTypes.function.isNode(node)) return _node_types.nodeTypes.function.toKqlExpression(node);
  if (_node_types.nodeTypes.literal.isNode(node)) return _node_types.nodeTypes.literal.toKqlExpression(node);
  if (_node_types.nodeTypes.wildcard.isNode(node)) return _node_types.nodeTypes.wildcard.toKqlExpression(node);
  throw new Error(`Unknown KQL node type: "${node.type}"`);
}

/**
 * @params {String} indexPattern
 * @params {Object} config - contains the dateFormatTZ
 *
 * IndexPattern isn't required, but if you pass one in, we can be more intelligent
 * about how we craft the queries (e.g. scripted fields)
 *
 */
const toElasticsearchQuery = (node, indexPattern, config = {}, context) => {
  if (!node || !node.type || !_node_types.nodeTypes[node.type]) {
    return toElasticsearchQuery(_node_types.nodeTypes.function.buildNode('and', []), indexPattern);
  }

  // TODO: the return type of this function might be incorrect and it works only because of this casting
  const nodeType = _node_types.nodeTypes[node.type];
  return nodeType.toElasticsearchQuery(node, indexPattern, config, context);
};
exports.toElasticsearchQuery = toElasticsearchQuery;