"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.unpinFilter = exports.toggleFilterPinned = exports.toggleFilterNegated = exports.toggleFilterDisabled = exports.pinFilter = exports.isFilters = exports.isFilterPinned = exports.isFilterDisabled = exports.isFilter = exports.enableFilter = exports.disableFilter = exports.cleanFilter = void 0;
var _lodash = require("lodash");
var _build_filters = require("../build_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 *
 * @param filter
 * @returns `true` if the filter should be applied to global scope
 *
 * @public
 */
const isFilterPinned = filter => {
  return filter.$state && filter.$state.store === _build_filters.FilterStateStore.GLOBAL_STATE;
};

/**
 * @param filter
 * @returns `true` if the filter is disabled
 *
 * @public
 */
exports.isFilterPinned = isFilterPinned;
const isFilterDisabled = filter => (0, _lodash.get)(filter, 'meta.disabled', false);

/**
 *
 * @param filter
 * @returns A copy of the filter with a toggled disabled state
 *
 * @public
 */
exports.isFilterDisabled = isFilterDisabled;
const toggleFilterDisabled = filter => {
  const disabled = !filter.meta.disabled;
  const meta = {
    ...filter.meta,
    disabled
  };
  return {
    ...filter,
    meta
  };
};

/**
 *
 * @param filter
 * @returns A copy of the filter with a toggled negated state
 *
 * @public
 */
exports.toggleFilterDisabled = toggleFilterDisabled;
const toggleFilterNegated = filter => {
  const negate = !filter.meta.negate;
  const meta = {
    ...filter.meta,
    negate
  };
  return {
    ...filter,
    meta
  };
};

/**
 *
 * @param filter
 * @returns A copy of the filter with a toggled pinned state (toggles store from app to global and vice versa)
 *
 * @public
 */
exports.toggleFilterNegated = toggleFilterNegated;
const toggleFilterPinned = filter => {
  const store = isFilterPinned(filter) ? _build_filters.FilterStateStore.APP_STATE : _build_filters.FilterStateStore.GLOBAL_STATE;
  const $state = {
    ...filter.$state,
    store
  };
  return {
    ...filter,
    $state
  };
};

/**
 * @param filter
 * @returns An enabled copy of the filter
 *
 * @public
 */
exports.toggleFilterPinned = toggleFilterPinned;
const enableFilter = filter => !filter.meta.disabled ? filter : toggleFilterDisabled(filter);

/**
 * @param filter
 * @returns A disabled copy of the filter
 *
 * @public
 */
exports.enableFilter = enableFilter;
const disableFilter = filter => filter.meta.disabled ? filter : toggleFilterDisabled(filter);

/**
 * @param filter
 * @returns A pinned (global) copy of the filter
 *
 * @public
 */
exports.disableFilter = disableFilter;
const pinFilter = filter => isFilterPinned(filter) ? filter : toggleFilterPinned(filter);

/**
 * @param filter
 * @returns An unpinned (app scoped) copy of the filter
 *
 * @public
 */
exports.pinFilter = pinFilter;
const unpinFilter = filter => !isFilterPinned(filter) ? filter : toggleFilterPinned(filter);

/**
 * @param {unknown} filter
 * @returns `true` if the given object is a filter
 *
 * @public
 */
exports.unpinFilter = unpinFilter;
const isFilter = x => !!x && typeof x === 'object' && !!x.meta && typeof x.meta === 'object';

/**
 * @param {unknown} filters
 * @returns `true` if the given object is an array of filters
 *
 * @public
 */
exports.isFilter = isFilter;
const isFilters = x => Array.isArray(x) && !x.find(y => !isFilter(y));

/**
 * Clean out decorators from the filters
 * @param {object} filter The filter to clean
 * @returns {object}
 *
 * @public
 */
exports.isFilters = isFilters;
const cleanFilter = filter => (0, _lodash.omit)(filter, ['meta', '$state']);
exports.cleanFilter = cleanFilter;