"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchMlModels = exports.fetchCompatiblePromotedModelIds = void 0;
var _i18n = require("@kbn/i18n");
var _ml_inference_pipeline = require("../../../common/ml_inference_pipeline");
var _ml = require("../../../common/types/ml");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

let compatibleElserModelId = _utils.ELSER_MODEL_ID;
let compatibleE5ModelId = _utils.E5_MODEL_ID;

/**
 * Fetches and enriches trained model information and deployment status. Pins promoted models (ELSER, E5) to the top. If a promoted model doesn't exist, a placeholder will be used.
 *
 * @param trainedModelsProvider Trained ML models provider
 * @returns List of models
 */
const fetchMlModels = async (trainedModelsProvider, log) => {
  if (!trainedModelsProvider) {
    throw new Error('Machine Learning is not enabled');
  }

  // Set the compatible ELSER and E5 model IDs based on platform architecture
  [compatibleElserModelId, compatibleE5ModelId] = await fetchCompatiblePromotedModelIds(trainedModelsProvider);

  // Get compatible variants of placeholder models
  const modelPlaceholders = [_utils.ELSER_MODEL_PLACEHOLDER, _utils.ELSER_LINUX_OPTIMIZED_MODEL_PLACEHOLDER, _utils.E5_MODEL_PLACEHOLDER, _utils.E5_LINUX_OPTIMIZED_MODEL_PLACEHOLDER].filter(model => isCompatiblePromotedModelId(model.modelId));

  // Fetch all models and their deployment stats using the ML client
  const modelsResponse = await trainedModelsProvider.getTrainedModels({});
  if (modelsResponse.count === 0) {
    return modelPlaceholders;
  }
  const modelsStatsResponse = await trainedModelsProvider.getTrainedModelsStats({});
  const fetchedModels = modelsResponse.trained_model_configs
  // Filter unsupported models
  .filter(modelConfig => isSupportedModel(modelConfig))
  // Get corresponding model stats and compose full model object
  .map(modelConfig => getModel(modelConfig, modelsStatsResponse.trained_model_stats.find(m => m.model_id === modelConfig.model_id)));

  // Merge fetched models with placeholders
  const mergedModels = mergeModels(modelPlaceholders, fetchedModels);

  // Undeployed placeholder models might be in the Downloading phase; let's evaluate this with a call
  // We must do this one by one because the API doesn't support fetching multiple models with include=definition_status
  const enrichedModelPromises = mergedModels.map(model => enrichModelWithDownloadStatus(model, trainedModelsProvider, log));
  const enrichedModels = await Promise.all(enrichedModelPromises);

  // Pin ELSER to the top, then E5 below, then the rest of the models sorted alphabetically
  return enrichedModels.sort(sortModels);
};

/**
 * Fetches model IDs of promoted models (ELSER, E5) that are compatible with the platform architecture. The fetches
 * are executed in parallel.
 * Defaults to the cross-platform variant of a model if its ID is not present in the trained models client's response.
 * @param trainedModelsProvider Trained ML models provider
 * @returns Array of model IDs [0: ELSER, 1: E5]
 */
exports.fetchMlModels = fetchMlModels;
const fetchCompatiblePromotedModelIds = async trainedModelsProvider => {
  var _compatibleModelConfi, _compatibleModelConfi2, _compatibleModelConfi3, _compatibleModelConfi4;
  const compatibleModelConfigs = await Promise.all([trainedModelsProvider.getCuratedModelConfig('elser', {
    version: 2
  }), trainedModelsProvider.getCuratedModelConfig('e5')]);
  return [(_compatibleModelConfi = (_compatibleModelConfi2 = compatibleModelConfigs.find(modelConfig => (modelConfig === null || modelConfig === void 0 ? void 0 : modelConfig.modelName) === 'elser')) === null || _compatibleModelConfi2 === void 0 ? void 0 : _compatibleModelConfi2.model_id) !== null && _compatibleModelConfi !== void 0 ? _compatibleModelConfi : _utils.ELSER_MODEL_ID, (_compatibleModelConfi3 = (_compatibleModelConfi4 = compatibleModelConfigs.find(modelConfig => (modelConfig === null || modelConfig === void 0 ? void 0 : modelConfig.modelName) === 'e5')) === null || _compatibleModelConfi4 === void 0 ? void 0 : _compatibleModelConfi4.model_id) !== null && _compatibleModelConfi3 !== void 0 ? _compatibleModelConfi3 : _utils.E5_MODEL_ID];
};
exports.fetchCompatiblePromotedModelIds = fetchCompatiblePromotedModelIds;
const getModel = (modelConfig, modelStats) => {
  const modelId = modelConfig.model_id;
  const type = modelConfig.inference_config ? Object.keys(modelConfig.inference_config)[0] : '';
  const model = {
    ..._utils.BASE_MODEL,
    modelId,
    type,
    title: getUserFriendlyTitle(modelId, type),
    description: modelConfig.description,
    types: (0, _ml_inference_pipeline.getMlModelTypesForModelConfig)(modelConfig),
    inputFieldNames: modelConfig.input.field_names,
    version: modelConfig.version,
    isPromoted: [_utils.ELSER_MODEL_ID, _utils.ELSER_LINUX_OPTIMIZED_MODEL_ID, _utils.E5_MODEL_ID, _utils.E5_LINUX_OPTIMIZED_MODEL_ID].includes(modelId)
  };

  // Enrich deployment stats
  if (modelStats && modelStats.deployment_stats) {
    model.hasStats = true;
    model.deploymentState = getDeploymentState(modelStats.deployment_stats.allocation_status.state);
    model.nodeAllocationCount = modelStats.deployment_stats.allocation_status.allocation_count;
    model.targetAllocationCount = modelStats.deployment_stats.allocation_status.target_allocation_count;
    model.threadsPerAllocation = modelStats.deployment_stats.threads_per_allocation;
    model.startTime = modelStats.deployment_stats.start_time;
  } else if (model.modelId === _utils.LANG_IDENT_MODEL_ID) {
    model.deploymentState = _ml.MlModelDeploymentState.FullyAllocated;
  }
  return model;
};
const enrichModelWithDownloadStatus = (model, trainedModelsProvider, log) => {
  // Only enrich promoted non-placeholder models
  if (!model.isPromoted || model.isPlaceholder || model.hasStats) {
    return Promise.resolve(model);
  }
  return trainedModelsProvider.getTrainedModels({
    model_id: model.modelId,
    include: 'definition_status'
  }).then(modelConfigWithDefinitionStatus => {
    // We're using NotDeployed for downloaded models. Downloaded is also a valid status, but we want to have the same
    // status badge as for 3rd party models.
    return {
      ...model,
      ...(modelConfigWithDefinitionStatus.count > 0 ? {
        deploymentState: modelConfigWithDefinitionStatus.trained_model_configs[0].fully_defined ? _ml.MlModelDeploymentState.NotDeployed : _ml.MlModelDeploymentState.Downloading
      } : {})
    };
  }).catch(err => {
    // Log and swallow error
    log.warn(`Failed to retrieve definition status of model ${model.modelId}: ${err}`);
    return model;
  });
};
const mergeModels = (modelPlaceholders, fetchedModels) => [
// Placeholder models that have no fetched model counterparts
...modelPlaceholders.filter(m => !fetchedModels.find(f => f.modelId === m.modelId)),
// Combined fetched and placeholder models
...fetchedModels.map(f => {
  const modelPlaceholder = modelPlaceholders.find(m => m.modelId === f.modelId);
  if (modelPlaceholder) {
    // Keep title, description and those properties from placeholder that are undefined in fetched model
    const {
      title,
      description,
      ...modelWithoutTitleAndDescription
    } = f;
    return Object.assign({}, modelPlaceholder, modelWithoutTitleAndDescription);
  }
  return f;
})];
const isCompatiblePromotedModelId = modelId => [compatibleElserModelId, compatibleE5ModelId].includes(modelId);

/**
 * A model is supported if:
 * - The inference type is supported, AND
 * - The model is the compatible variant of ELSER/E5, or it's a 3rd party model
 */
const isSupportedModel = modelConfig => isSupportedInferenceType(modelConfig) && (!modelConfig.model_id.startsWith(_utils.ELSER_MODEL_ID) && !modelConfig.model_id.startsWith(_utils.E5_MODEL_ID) || isCompatiblePromotedModelId(modelConfig.model_id));
const isSupportedInferenceType = modelConfig => Object.keys(modelConfig.inference_config || {}).some(inferenceType => Object.keys(_utils.MODEL_TITLES_BY_TYPE).includes(inferenceType)) || modelConfig.model_id === _utils.LANG_IDENT_MODEL_ID;

/**
 * Sort function for models; makes ELSER go to the top, then E5, then the rest of the models sorted by title.
 */
const sortModels = (m1, m2) => m1.modelId.startsWith(_utils.ELSER_MODEL_ID) ? -1 : m2.modelId.startsWith(_utils.ELSER_MODEL_ID) ? 1 : m1.modelId.startsWith(_utils.E5_MODEL_ID) ? -1 : m2.modelId.startsWith(_utils.E5_MODEL_ID) ? 1 : m1.title.localeCompare(m2.title);
const getUserFriendlyTitle = (modelId, modelType) => {
  return _utils.MODEL_TITLES_BY_TYPE[modelType] !== undefined ? _utils.MODEL_TITLES_BY_TYPE[modelType] : modelId === _utils.LANG_IDENT_MODEL_ID ? _i18n.i18n.translate('xpack.enterpriseSearch.content.ml_inference.lang_ident', {
    defaultMessage: 'Language Identification'
  }) : modelId;
};
const getDeploymentState = state => {
  switch (state) {
    case 'starting':
      return _ml.MlModelDeploymentState.Starting;
    case 'started':
      return _ml.MlModelDeploymentState.Started;
    case 'fully_allocated':
      return _ml.MlModelDeploymentState.FullyAllocated;
  }
  return _ml.MlModelDeploymentState.NotDeployed;
};