"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EngineLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _constants = require("../credentials/constants");
var _constants2 = require("./constants");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EngineLogic = exports.EngineLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'engine_logic'],
  actions: {
    setEngineData: engine => ({
      engine
    }),
    setEngineName: engineName => ({
      engineName
    }),
    setEngineNotFound: notFound => ({
      notFound
    }),
    clearEngine: true,
    initializeEngine: true,
    pollEmptyEngine: true,
    onPollStart: intervalId => ({
      intervalId
    }),
    stopPolling: true,
    onPollStop: true
  },
  reducers: {
    dataLoading: [true, {
      setEngineData: () => false,
      clearEngine: () => true
    }],
    engine: [{}, {
      setEngineData: (_, {
        engine
      }) => engine,
      clearEngine: () => ({})
    }],
    engineName: ['', {
      setEngineName: (_, {
        engineName
      }) => engineName,
      clearEngine: () => ''
    }],
    engineNotFound: [false, {
      setEngineNotFound: (_, {
        notFound
      }) => notFound,
      clearEngine: () => false
    }],
    intervalId: [null, {
      onPollStart: (_, {
        intervalId
      }) => intervalId,
      onPollStop: () => null
    }]
  },
  selectors: ({
    selectors
  }) => ({
    hasNoDocuments: [() => [selectors.engine], engine => !engine.document_count],
    hasEmptySchema: [() => [selectors.engine], engine => Object.keys(engine.schema || {}).length === 0],
    isMetaEngine: [() => [selectors.engine], engine => (engine === null || engine === void 0 ? void 0 : engine.type) === _types.EngineTypes.meta],
    isElasticsearchEngine: [() => [selectors.engine], engine => (engine === null || engine === void 0 ? void 0 : engine.type) === _types.EngineTypes.elasticsearch],
    isSampleEngine: [() => [selectors.engine], engine => !!(engine !== null && engine !== void 0 && engine.sample)],
    // Indexed engines
    hasSchemaErrors: [() => [selectors.engine], ({
      activeReindexJob
    }) => (activeReindexJob === null || activeReindexJob === void 0 ? void 0 : activeReindexJob.numDocumentsWithErrors) > 0],
    // Meta engines
    hasSchemaConflicts: [() => [selectors.engine], engine => !!(engine !== null && engine !== void 0 && engine.schemaConflicts && Object.keys(engine.schemaConflicts).length > 0)],
    hasUnconfirmedSchemaFields: [() => [selectors.engine], engine => {
      var _engine$unconfirmedFi;
      return (engine === null || engine === void 0 ? void 0 : (_engine$unconfirmedFi = engine.unconfirmedFields) === null || _engine$unconfirmedFi === void 0 ? void 0 : _engine$unconfirmedFi.length) > 0;
    }],
    searchKey: [() => [selectors.engine], engine => {
      const isSearchKey = token => token.type === _constants.ApiTokenTypes.Search;
      const searchKey = (engine.apiTokens || []).find(isSearchKey);
      return (searchKey === null || searchKey === void 0 ? void 0 : searchKey.key) || '';
    }],
    hasIncompleteFields: [() => [selectors.engine], ({
      incompleteFields
    }) => (incompleteFields === null || incompleteFields === void 0 ? void 0 : incompleteFields.length) > 0]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    initializeEngine: async (_, breakpoint) => {
      breakpoint(); // Prevents errors if logic unmounts while fetching

      const {
        engineName
      } = values;
      const {
        http
      } = _http.HttpLogic.values;
      try {
        const response = await http.get(`/internal/app_search/engines/${engineName}`);
        actions.setEngineData(response);
      } catch (error) {
        var _error$response, _error$response2;
        if ((error === null || error === void 0 ? void 0 : (_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.status) >= 400 && (error === null || error === void 0 ? void 0 : (_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : _error$response2.status) < 500) {
          actions.setEngineNotFound(true);
        } else {
          (0, _flash_messages.flashErrorToast)(_constants2.POLLING_ERROR_TITLE, {
            text: _constants2.POLLING_ERROR_TEXT,
            toastLifeTimeMs: _constants2.POLLING_DURATION * 0.75
          });
        }
      }
    },
    pollEmptyEngine: () => {
      if (values.intervalId) return; // Ensure we only have one poll at a time

      const id = window.setInterval(() => {
        if (values.hasNoDocuments) {
          actions.initializeEngine(); // Re-fetch engine data when engine is empty
        } else {
          actions.stopPolling();
        }
      }, _constants2.POLLING_DURATION);
      actions.onPollStart(id);
    },
    stopPolling: () => {
      if (values.intervalId !== null) {
        clearInterval(values.intervalId);
        actions.onPollStop();
      }
    }
  }),
  events: ({
    actions
  }) => ({
    beforeUnmount: () => {
      actions.stopPolling();
    }
  })
});