"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerReactEmbeddableFactory = exports.reactEmbeddableRegistryHasKey = exports.getReactEmbeddableFactory = void 0;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const registry = {};

/**
 * Registers a new React embeddable factory. This should be called at plugin start time.
 *
 * @param type The key to register the factory under. This should be the same as the `type` key in the factory definition.
 * @param getFactory an async function that gets the factory definition for this key. This should always async import the
 * actual factory definition file to avoid polluting page load.
 */
const registerReactEmbeddableFactory = (type, getFactory) => {
  if (registry[type] !== undefined) throw new Error(_i18n.i18n.translate('embeddableApi.reactEmbeddable.factoryAlreadyExistsError', {
    defaultMessage: 'An embeddable factory for type: {key} is already registered.',
    values: {
      key: type
    }
  }));
  registry[type] = getFactory;
};
exports.registerReactEmbeddableFactory = registerReactEmbeddableFactory;
const reactEmbeddableRegistryHasKey = key => registry[key] !== undefined;
exports.reactEmbeddableRegistryHasKey = reactEmbeddableRegistryHasKey;
const getReactEmbeddableFactory = async key => {
  if (registry[key] === undefined) throw new Error(_i18n.i18n.translate('embeddableApi.reactEmbeddable.factoryNotFoundError', {
    defaultMessage: 'No embeddable factory found for type: {key}',
    values: {
      key
    }
  }));
  return registry[key]();
};
exports.getReactEmbeddableFactory = getReactEmbeddableFactory;