"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postActionsConnectorExecuteRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _server = require("@kbn/data-plugin/server");
var _configSchema = require("@kbn/config-schema");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _i18n = require("@kbn/i18n");
var _utils = require("./utils");
var _event_based_telemetry = require("../lib/telemetry/event_based_telemetry");
var _executor = require("../lib/executor");
var _constants = require("../../common/constants");
var _helpers = require("../lib/langchain/helpers");
var _build_response = require("../lib/build_response");
var _constants2 = require("./knowledge_base/constants");
var _execute_custom_llm_chain = require("../lib/langchain/execute_custom_llm_chain");
var _helpers2 = require("./helpers");
var _utils2 = require("./evaluate/utils");
var _helpers3 = require("../ai_assistant_data_clients/anonymization_fields/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const postActionsConnectorExecuteRoute = (router, getElser) => {
  router.versioned.post({
    access: 'internal',
    path: _constants.POST_ACTIONS_CONNECTOR_EXECUTE,
    options: {
      tags: ['access:elasticAssistant']
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.ExecuteConnectorRequestBody),
        params: _configSchema.schema.object({
          connectorId: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, response) => {
    const abortSignal = (0, _server.getRequestAbortedSignal)(request.events.aborted$);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    let onLlmResponse;
    try {
      var _ref, _prevMessages3, _request$body$isEnabl;
      const authenticatedUser = assistantContext.getCurrentUser();
      if (authenticatedUser == null) {
        return response.unauthorized({
          body: `Authenticated user not found`
        });
      }
      const conversationsDataClient = await assistantContext.getAIAssistantConversationsDataClient();
      const anonymizationFieldsDataClient = await assistantContext.getAIAssistantAnonymizationFieldsDataClient();
      let latestReplacements = request.body.replacements;
      const onNewReplacements = newReplacements => {
        latestReplacements = {
          ...latestReplacements,
          ...newReplacements
        };
      };
      let prevMessages;
      let newMessage;
      const conversationId = request.body.conversationId;
      const actionTypeId = request.body.actionTypeId;
      const langSmithProject = request.body.langSmithProject;
      const langSmithApiKey = request.body.langSmithApiKey;

      // if message is undefined, it means the user is regenerating a message from the stored conversation
      if (request.body.message) {
        newMessage = {
          content: request.body.message,
          role: 'user'
        };
      }
      const connectorId = decodeURIComponent(request.params.connectorId);

      // get the actions plugin start contract from the request context:
      const actions = (await context.elasticAssistant).actions;
      if (conversationId) {
        var _conversation$message;
        const conversation = await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.getConversation({
          id: conversationId,
          authenticatedUser
        }));
        if (conversation == null) {
          return response.notFound({
            body: `conversation id: "${conversationId}" not found`
          });
        }

        // messages are anonymized by conversationsDataClient
        prevMessages = conversation === null || conversation === void 0 ? void 0 : (_conversation$message = conversation.messages) === null || _conversation$message === void 0 ? void 0 : _conversation$message.map(c => ({
          role: c.role,
          content: c.content
        }));
        if (request.body.message) {
          const res = await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.appendConversationMessages({
            existingConversation: conversation,
            messages: [{
              ...{
                content: (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
                  messageContent: request.body.message,
                  replacements: request.body.replacements
                }),
                role: 'user'
              },
              timestamp: new Date().toISOString()
            }]
          }));
          if (res == null) {
            return response.badRequest({
              body: `conversation id: "${conversationId}" not updated`
            });
          }
        }
        const updatedConversation = await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.getConversation({
          id: conversationId,
          authenticatedUser
        }));
        if (updatedConversation == null) {
          return response.notFound({
            body: `conversation id: "${conversationId}" not found`
          });
        }
        const NEW_CHAT = _i18n.i18n.translate('xpack.elasticAssistantPlugin.server.newChat', {
          defaultMessage: 'New chat'
        });
        if ((conversation === null || conversation === void 0 ? void 0 : conversation.title) === NEW_CHAT && prevMessages) {
          try {
            var _newMessage, _prevMessages;
            const autoTitle = await (0, _executor.executeAction)({
              actions,
              request,
              connectorId,
              actionTypeId,
              params: {
                subAction: 'invokeAI',
                subActionParams: {
                  model: request.body.model,
                  messages: [{
                    role: 'system',
                    content: _i18n.i18n.translate('xpack.elasticAssistantPlugin.server.autoTitlePromptDescription', {
                      defaultMessage: 'You are a helpful assistant for Elastic Security. Assume the following message is the start of a conversation between you and a user; give this conversation a title based on the content below. DO NOT UNDER ANY CIRCUMSTANCES wrap this title in single or double quotes. This title is shown in a list of conversations to the user, so title it for the user, not for you.'
                    })
                  }, (_newMessage = newMessage) !== null && _newMessage !== void 0 ? _newMessage : (_prevMessages = prevMessages) === null || _prevMessages === void 0 ? void 0 : _prevMessages[0]],
                  ...(actionTypeId === '.gen-ai' ? {
                    n: 1,
                    stop: null,
                    temperature: 0.2
                  } : {
                    temperature: 0,
                    stopSequences: []
                  })
                }
              },
              logger
            }); // TODO: Use function overloads in executeAction to avoid this cast when sending subAction: 'invokeAI',
            if (autoTitle.status === 'ok') {
              try {
                // This regular expression captures a string enclosed in single or double quotes.
                // It extracts the string content without the quotes.
                // Example matches:
                // - "Hello, World!" => Captures: Hello, World!
                // - 'Another Example' => Captures: Another Example
                // - JustTextWithoutQuotes => Captures: JustTextWithoutQuotes
                const match = autoTitle.data.match(/^["']?([^"']+)["']?$/);
                const title = match ? match[1] : autoTitle.data;
                await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.updateConversation({
                  conversationUpdateProps: {
                    id: conversationId,
                    title
                  }
                }));
              } catch (e) {
                logger.warn(`Failed to update conversation with generated title: ${e.message}`);
              }
            }
          } catch (e) {
            /* empty */
          }
        }
        onLlmResponse = async (content, traceData = {}, isError = false) => {
          if (updatedConversation) {
            await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.appendConversationMessages({
              existingConversation: updatedConversation,
              messages: [(0, _helpers2.getMessageFromRawResponse)({
                rawContent: (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
                  messageContent: content,
                  replacements: latestReplacements
                }),
                traceData,
                isError
              })]
            }));
          }
          if (Object.keys(latestReplacements).length > 0) {
            await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.updateConversation({
              conversationUpdateProps: {
                id: conversationId,
                replacements: latestReplacements
              }
            }));
          }
        };
      }

      // if not langchain, call execute action directly and return the response:
      if (!request.body.isEnabledKnowledgeBase && !request.body.isEnabledRAGAlerts) {
        var _prevMessages2;
        logger.debug('Executing via actions framework directly');
        const result = await (0, _executor.executeAction)({
          abortSignal,
          onLlmResponse,
          actions,
          request,
          connectorId,
          actionTypeId,
          params: {
            subAction: request.body.subAction,
            subActionParams: {
              model: request.body.model,
              messages: [...((_prevMessages2 = prevMessages) !== null && _prevMessages2 !== void 0 ? _prevMessages2 : []), ...(newMessage ? [newMessage] : [])],
              ...(actionTypeId === '.gen-ai' ? {
                n: 1,
                stop: null,
                temperature: 0.2
              } : {
                temperature: 0,
                stopSequences: []
              })
            }
          },
          logger
        });
        telemetry.reportEvent(_event_based_telemetry.INVOKE_ASSISTANT_SUCCESS_EVENT.eventType, {
          actionTypeId,
          isEnabledKnowledgeBase: request.body.isEnabledKnowledgeBase,
          isEnabledRAGAlerts: request.body.isEnabledRAGAlerts,
          model: request.body.model,
          assistantStreamingEnabled: request.body.subAction !== 'invokeAI'
        });
        return response.ok({
          body: result
        });
      }

      // TODO: Add `traceId` to actions request when calling via langchain
      logger.debug(`Executing via langchain, isEnabledKnowledgeBase: ${request.body.isEnabledKnowledgeBase}, isEnabledRAGAlerts: ${request.body.isEnabledRAGAlerts}`);

      // Fetch any tools registered by the request's originating plugin
      const pluginName = (0, _helpers2.getPluginNameFromRequest)({
        request,
        defaultPluginName: _helpers2.DEFAULT_PLUGIN_NAME,
        logger
      });
      const assistantTools = (await context.elasticAssistant).getRegisteredTools(pluginName).filter(x => x.id !== 'attack-discovery'); // We don't (yet) support asking the assistant for NEW attack discoveries from a conversation

      // get a scoped esClient for assistant memory
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;

      // convert the assistant messages to LangChain messages:
      const langChainMessages = (0, _helpers.getLangChainMessages)((_ref = [...((_prevMessages3 = prevMessages) !== null && _prevMessages3 !== void 0 ? _prevMessages3 : []), ...(newMessage ? [newMessage] : [])]) !== null && _ref !== void 0 ? _ref : []);
      const elserId = await getElser(request, (await context.core).savedObjects.getClient());
      const anonymizationFieldsRes = await (anonymizationFieldsDataClient === null || anonymizationFieldsDataClient === void 0 ? void 0 : anonymizationFieldsDataClient.findDocuments({
        perPage: 1000,
        page: 1
      }));
      const result = await (0, _execute_custom_llm_chain.callAgentExecutor)({
        abortSignal,
        alertsIndexPattern: request.body.alertsIndexPattern,
        anonymizationFields: anonymizationFieldsRes ? (0, _helpers3.transformESSearchToAnonymizationFields)(anonymizationFieldsRes.data) : undefined,
        actions,
        isEnabledKnowledgeBase: (_request$body$isEnabl = request.body.isEnabledKnowledgeBase) !== null && _request$body$isEnabl !== void 0 ? _request$body$isEnabl : false,
        assistantTools,
        connectorId,
        elserId,
        esClient,
        isStream:
        // TODO implement llmClass for bedrock streaming
        // tracked here: https://github.com/elastic/security-team/issues/7363
        request.body.subAction !== 'invokeAI' && actionTypeId === '.gen-ai',
        llmType: (0, _utils.getLlmType)(actionTypeId),
        kbResource: _constants2.ESQL_RESOURCE,
        langChainMessages,
        logger,
        onNewReplacements,
        onLlmResponse,
        request,
        replacements: request.body.replacements,
        size: request.body.size,
        telemetry,
        traceOptions: {
          projectName: langSmithProject,
          tracers: (0, _utils2.getLangSmithTracer)({
            apiKey: langSmithApiKey,
            projectName: langSmithProject,
            logger
          })
        }
      });
      telemetry.reportEvent(_event_based_telemetry.INVOKE_ASSISTANT_SUCCESS_EVENT.eventType, {
        actionTypeId,
        isEnabledKnowledgeBase: request.body.isEnabledKnowledgeBase,
        isEnabledRAGAlerts: request.body.isEnabledRAGAlerts,
        model: request.body.model,
        // TODO rm actionTypeId check when llmClass for bedrock streaming is implemented
        // tracked here: https://github.com/elastic/security-team/issues/7363
        assistantStreamingEnabled: request.body.subAction !== 'invokeAI' && actionTypeId === '.gen-ai'
      });
      return response.ok(result);
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      if (onLlmResponse) {
        onLlmResponse(error.message, {}, true);
      }
      telemetry.reportEvent(_event_based_telemetry.INVOKE_ASSISTANT_ERROR_EVENT.eventType, {
        actionTypeId: request.body.actionTypeId,
        isEnabledKnowledgeBase: request.body.isEnabledKnowledgeBase,
        isEnabledRAGAlerts: request.body.isEnabledRAGAlerts,
        model: request.body.model,
        errorMessage: error.message,
        assistantStreamingEnabled: request.body.subAction !== 'invokeAI'
      });
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.postActionsConnectorExecuteRoute = postActionsConnectorExecuteRoute;