"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postKnowledgeBaseRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _build_response = require("../../lib/build_response");
var _constants = require("../../../common/constants");
var _elasticsearch_store = require("../../lib/langchain/elasticsearch_store/elasticsearch_store");
var _constants2 = require("./constants");
var _get_kb_resource = require("./get_kb_resource");
var _esql_loader = require("../../lib/langchain/content_loaders/esql_loader");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Load Knowledge Base index, pipeline, and resources (collection of documents)
 * @param router
 */
const postKnowledgeBaseRoute = (router, getElser) => {
  router.versioned.post({
    access: 'internal',
    path: _constants.KNOWLEDGE_BASE,
    options: {
      // Note: Relying on current user privileges to scope an esClient.
      // Add `access:kbnElasticAssistant` to limit API access to only users with assistant privileges
      tags: []
    }
  }).addVersion({
    version: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_INTERNAL_API_VERSION,
    validate: {
      request: {
        params: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.CreateKnowledgeBaseRequestParams)
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    try {
      const core = await context.core;
      // Get a scoped esClient for creating the Knowledge Base index, pipeline, and documents
      const esClient = core.elasticsearch.client.asCurrentUser;
      const elserId = await getElser(request, core.savedObjects.getClient());
      const kbResource = (0, _get_kb_resource.getKbResource)(request);
      const esStore = new _elasticsearch_store.ElasticsearchStore(esClient, _constants2.KNOWLEDGE_BASE_INDEX_PATTERN, logger, telemetry, elserId, kbResource);

      // Pre-check on index/pipeline
      let indexExists = await esStore.indexExists();
      let pipelineExists = await esStore.pipelineExists();

      // Load if not exists
      if (!pipelineExists) {
        pipelineExists = await esStore.createPipeline();
      }
      if (!indexExists) {
        indexExists = await esStore.createIndex();
      }

      // If specific resource is requested, load it
      if (kbResource === _constants2.ESQL_RESOURCE) {
        const esqlExists = (await esStore.similaritySearch(_constants2.ESQL_DOCS_LOADED_QUERY)).length > 0;
        if (!esqlExists) {
          const loadedKnowledgeBase = await (0, _esql_loader.loadESQL)(esStore, logger);
          return response.custom({
            body: {
              success: loadedKnowledgeBase
            },
            statusCode: 201
          });
        } else {
          return response.ok({
            body: {
              success: true
            }
          });
        }
      }
      const wasSuccessful = indexExists && pipelineExists;
      if (wasSuccessful) {
        return response.ok({
          body: {
            success: true
          }
        });
      } else {
        return response.custom({
          body: {
            success: false
          },
          statusCode: 500
        });
      }
    } catch (err) {
      logger.log(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.postKnowledgeBaseRoute = postKnowledgeBaseRoute;