"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.APMTracer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _base = require("@langchain/core/tracers/base");
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * APMTracer is a tracer that uses the Elastic APM agent to trace langchain retrievers, llms, chains, and tools.
 */
class APMTracer extends _base.BaseTracer {
  constructor(fields = {}, logger) {
    super(fields);
    (0, _defineProperty2.default)(this, "name", 'apm_tracer');
    (0, _defineProperty2.default)(this, "projectName", void 0);
    (0, _defineProperty2.default)(this, "exampleId", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "retrieverSpans", []);
    (0, _defineProperty2.default)(this, "llmSpans", []);
    (0, _defineProperty2.default)(this, "chainSpans", []);
    (0, _defineProperty2.default)(this, "toolSpans", []);
    const {
      exampleId,
      projectName
    } = fields;
    this.projectName = projectName !== null && projectName !== void 0 ? projectName : 'default';
    this.exampleId = exampleId;
    this.logger = logger;
  }
  async persistRun(_run) {}

  /**
   * LangChain Run's contain a lot of useful information, so here we unpack as much of it as we can
   * into labels that can be added to the corresponding span. Stringifying outputs at the moment since
   * the Run schema is a loose KVMap, but we should more elegantly unpack relevant data that we find useful
   *
   * See BaseRun interface Run extends from
   *
   * @param run
   * @protected
   */
  _getLabelsFromRun(run) {
    try {
      return {
        tags: JSON.stringify(run.tags),
        outputs: JSON.stringify(run.outputs),
        events: JSON.stringify(run.events),
        inputs: JSON.stringify(run.inputs)
      };
    } catch (e) {
      this.logger.error(`Error parsing run into labels:\n${e}`);
      return {};
    }
  }
  createAndAddSpanFromRun(run, spans) {
    var _agent$startSpan;
    const span = (_agent$startSpan = _elasticApmNode.default.startSpan(run.name)) !== null && _agent$startSpan !== void 0 ? _agent$startSpan : undefined;
    if (span) {
      span.addLabels(this._getLabelsFromRun(run));
      spans.push(span);
    }
  }
  async onRetrieverStart(run) {
    this.logger.debug(`onRetrieverStart: run:\n${JSON.stringify(run, null, 2)}`);
    this.createAndAddSpanFromRun(run, this.retrieverSpans);
  }
  async onRetrieverEnd(run) {
    this.logger.debug(`onRetrieverEnd: run:\n${JSON.stringify(run, null, 2)}`);
    const span = this.retrieverSpans.pop();
    if (span != null) {
      span.addLabels(this._getLabelsFromRun(run));
      span.end();
    }
  }
  async onRetrieverError(run) {
    this.logger.debug(`onRetrieverError: run:\n${JSON.stringify(run, null, 2)}`);
  }
  async onLLMStart(run) {
    this.logger.debug(`onLLMStart: run:\n${JSON.stringify(run, null, 2)}`);
    this.createAndAddSpanFromRun(run, this.llmSpans);
  }
  async onLLMEnd(run) {
    this.logger.debug(`onLLMEnd: run:\n${JSON.stringify(run, null, 2)}`);
    const span = this.llmSpans.pop();
    if (span != null) {
      span.addLabels(this._getLabelsFromRun(run));
      span.end();
    }
  }
  async onLLMError(run) {
    this.logger.debug(`onLLMError: run:\n${JSON.stringify(run, null, 2)}`);
  }
  async onChainStart(run) {
    this.logger.debug(`onChainStart: run:\n${JSON.stringify(run, null, 2)}`);
    this.createAndAddSpanFromRun(run, this.chainSpans);
  }
  async onChainEnd(run) {
    this.logger.debug(`onChainEnd: run:\n${JSON.stringify(run, null, 2)}`);
    const span = this.chainSpans.pop();
    if (span != null) {
      span.addLabels(this._getLabelsFromRun(run));
      span.end();
    }
  }
  async onChainError(run) {
    this.logger.debug(`onChainError: run:\n${JSON.stringify(run, null, 2)}`);
  }
  async onToolStart(run) {
    this.logger.debug(`onToolStart: run:\n${JSON.stringify(run, null, 2)}`);
    this.createAndAddSpanFromRun(run, this.toolSpans);
  }
  async onToolEnd(run) {
    this.logger.debug(`onToolEnd: run:\n${JSON.stringify(run, null, 2)}`);
    const span = this.toolSpans.pop();
    if (span != null) {
      span.addLabels(this._getLabelsFromRun(run));
      span.end();
    }
  }
  async onToolError(run) {
    this.logger.debug(`onToolError: run:\n${JSON.stringify(run, null, 2)}`);
  }
}
exports.APMTracer = APMTracer;