"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AIAssistantService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _dataStreamAdapter = require("@kbn/data-stream-adapter");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _anonymization = require("../../common/anonymization");
var _conversations = require("../ai_assistant_data_clients/conversations");
var _create_resource_installation_helper = require("./create_resource_installation_helper");
var _field_maps_configuration = require("../ai_assistant_data_clients/conversations/field_maps_configuration");
var _field_maps_configuration2 = require("../ai_assistant_data_clients/prompts/field_maps_configuration");
var _field_maps_configuration3 = require("../ai_assistant_data_clients/anonymization_fields/field_maps_configuration");
var _ai_assistant_data_clients = require("../ai_assistant_data_clients");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TOTAL_FIELDS_LIMIT = 2500;
function getResourceName(resource) {
  return `.kibana-elastic-ai-assistant-${resource}`;
}
class AIAssistantService {
  constructor(options) {
    (0, _defineProperty2.default)(this, "initialized", void 0);
    (0, _defineProperty2.default)(this, "isInitializing", false);
    (0, _defineProperty2.default)(this, "conversationsDataStream", void 0);
    (0, _defineProperty2.default)(this, "promptsDataStream", void 0);
    (0, _defineProperty2.default)(this, "anonymizationFieldsDataStream", void 0);
    (0, _defineProperty2.default)(this, "resourceInitializationHelper", void 0);
    (0, _defineProperty2.default)(this, "initPromise", void 0);
    (0, _defineProperty2.default)(this, "createDataStream", ({
      resource,
      kibanaVersion,
      fieldMap
    }) => {
      const newDataStream = new _dataStreamAdapter.DataStreamSpacesAdapter(this.resourceNames.aliases[resource], {
        kibanaVersion,
        totalFieldsLimit: TOTAL_FIELDS_LIMIT
      });
      newDataStream.setComponentTemplate({
        name: this.resourceNames.componentTemplate[resource],
        fieldMap
      });
      newDataStream.setIndexTemplate({
        name: this.resourceNames.indexTemplate[resource],
        componentTemplateRefs: [this.resourceNames.componentTemplate[resource]]
      });
      return newDataStream;
    });
    (0, _defineProperty2.default)(this, "resourceNames", {
      componentTemplate: {
        conversations: getResourceName('component-template-conversations'),
        prompts: getResourceName('component-template-prompts'),
        anonymizationFields: getResourceName('component-template-anonymization-fields'),
        kb: getResourceName('component-template-kb')
      },
      aliases: {
        conversations: getResourceName('conversations'),
        prompts: getResourceName('prompts'),
        anonymizationFields: getResourceName('anonymization-fields'),
        kb: getResourceName('kb')
      },
      indexPatterns: {
        conversations: getResourceName('conversations*'),
        prompts: getResourceName('prompts*'),
        anonymizationFields: getResourceName('anonymization-fields*'),
        kb: getResourceName('kb*')
      },
      indexTemplate: {
        conversations: getResourceName('index-template-conversations'),
        prompts: getResourceName('index-template-prompts'),
        anonymizationFields: getResourceName('index-template-anonymization-fields'),
        kb: getResourceName('index-template-kb')
      },
      pipelines: {
        kb: getResourceName('kb-ingest-pipeline')
      }
    });
    this.options = options;
    this.initialized = false;
    this.conversationsDataStream = this.createDataStream({
      resource: 'conversations',
      kibanaVersion: options.kibanaVersion,
      fieldMap: _field_maps_configuration.conversationsFieldMap
    });
    this.promptsDataStream = this.createDataStream({
      resource: 'prompts',
      kibanaVersion: options.kibanaVersion,
      fieldMap: _field_maps_configuration2.assistantPromptsFieldMap
    });
    this.anonymizationFieldsDataStream = this.createDataStream({
      resource: 'anonymizationFields',
      kibanaVersion: options.kibanaVersion,
      fieldMap: _field_maps_configuration3.assistantAnonymizationFieldsFieldMap
    });
    this.initPromise = this.initializeResources();
    this.resourceInitializationHelper = (0, _create_resource_installation_helper.createResourceInstallationHelper)(this.options.logger, this.initPromise, this.installAndUpdateSpaceLevelResources.bind(this));
  }
  isInitialized() {
    return this.initialized;
  }
  async initializeResources() {
    this.isInitializing = true;
    try {
      this.options.logger.debug(`Initializing resources for AIAssistantService`);
      const esClient = await this.options.elasticsearchClientPromise;
      await this.conversationsDataStream.install({
        esClient,
        logger: this.options.logger,
        pluginStop$: this.options.pluginStop$
      });
      await this.promptsDataStream.install({
        esClient,
        logger: this.options.logger,
        pluginStop$: this.options.pluginStop$
      });
      await this.anonymizationFieldsDataStream.install({
        esClient,
        logger: this.options.logger,
        pluginStop$: this.options.pluginStop$
      });
    } catch (error) {
      this.options.logger.error(`Error initializing AI assistant resources: ${error.message}`);
      this.initialized = false;
      this.isInitializing = false;
      return (0, _create_resource_installation_helper.errorResult)(error.message);
    }
    this.initialized = true;
    this.isInitializing = false;
    return (0, _create_resource_installation_helper.successResult)();
  }
  async checkResourcesInstallation(opts) {
    // Check if resources installation has succeeded
    const {
      result: initialized,
      error
    } = await this.getSpaceResourcesInitializationPromise(opts.spaceId);

    // If space evel resources initialization failed, retry
    if (!initialized && error) {
      var _opts$spaceId;
      let initRetryPromise;

      // If !this.initialized, we know that resource initialization failed
      // and we need to retry this before retrying the spaceId specific resources
      if (!this.initialized) {
        if (!this.isInitializing) {
          this.options.logger.info(`Retrying common resource initialization`);
          initRetryPromise = this.initializeResources();
        } else {
          this.options.logger.info(`Skipped retrying common resource initialization because it is already being retried.`);
        }
      }
      this.resourceInitializationHelper.retry(opts.spaceId, initRetryPromise);
      const retryResult = await this.resourceInitializationHelper.getInitializedResources((_opts$spaceId = opts.spaceId) !== null && _opts$spaceId !== void 0 ? _opts$spaceId : _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING);
      if (!retryResult.result) {
        const errorLogPrefix = `There was an error in the framework installing spaceId-level resources and creating concrete indices for spaceId "${opts.spaceId}" - `;
        // Retry also failed
        this.options.logger.warn(retryResult.error && error ? `${errorLogPrefix}Retry failed with errors: ${error}` : `${errorLogPrefix}Original error: ${error}; Error after retry: ${retryResult.error}`);
        return null;
      } else {
        this.options.logger.info(`Resource installation for "${opts.spaceId}" succeeded after retry`);
      }
    }
  }
  async createAIAssistantConversationsDataClient(opts) {
    const res = await this.checkResourcesInstallation(opts);
    if (res === null) {
      return null;
    }
    return new _conversations.AIAssistantConversationsDataClient({
      logger: this.options.logger,
      elasticsearchClientPromise: this.options.elasticsearchClientPromise,
      spaceId: opts.spaceId,
      kibanaVersion: this.options.kibanaVersion,
      indexPatternsResorceName: this.resourceNames.aliases.conversations,
      currentUser: opts.currentUser
    });
  }
  async createAIAssistantPromptsDataClient(opts) {
    const res = await this.checkResourcesInstallation(opts);
    if (res === null) {
      return null;
    }
    return new _ai_assistant_data_clients.AIAssistantDataClient({
      logger: this.options.logger,
      elasticsearchClientPromise: this.options.elasticsearchClientPromise,
      spaceId: opts.spaceId,
      kibanaVersion: this.options.kibanaVersion,
      indexPatternsResorceName: this.resourceNames.aliases.prompts,
      currentUser: opts.currentUser
    });
  }
  async createAIAssistantAnonymizationFieldsDataClient(opts) {
    const res = await this.checkResourcesInstallation(opts);
    if (res === null) {
      return null;
    }
    return new _ai_assistant_data_clients.AIAssistantDataClient({
      logger: this.options.logger,
      elasticsearchClientPromise: this.options.elasticsearchClientPromise,
      spaceId: opts.spaceId,
      kibanaVersion: this.options.kibanaVersion,
      indexPatternsResorceName: this.resourceNames.aliases.anonymizationFields,
      currentUser: opts.currentUser
    });
  }
  async getSpaceResourcesInitializationPromise(spaceId = _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING) {
    const result = await this.resourceInitializationHelper.getInitializedResources(spaceId);
    // If the spaceId is unrecognized and spaceId is not the default, we
    // need to kick off resource installation and return the promise
    if (result.error && result.error.includes(`Unrecognized spaceId`) && spaceId !== _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING) {
      this.resourceInitializationHelper.add(spaceId);
      return this.resourceInitializationHelper.getInitializedResources(spaceId);
    }
    return result;
  }
  async installAndUpdateSpaceLevelResources(spaceId = _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING) {
    try {
      this.options.logger.debug(`Initializing spaceId level resources for AIAssistantService`);
      const conversationsIndexName = await this.conversationsDataStream.getInstalledSpaceName(spaceId);
      if (!conversationsIndexName) {
        await this.conversationsDataStream.installSpace(spaceId);
      }
      const promptsIndexName = await this.promptsDataStream.getInstalledSpaceName(spaceId);
      if (!promptsIndexName) {
        await this.promptsDataStream.installSpace(spaceId);
      }
      const anonymizationFieldsIndexName = await this.anonymizationFieldsDataStream.getInstalledSpaceName(spaceId);
      if (!anonymizationFieldsIndexName) {
        await this.anonymizationFieldsDataStream.installSpace(spaceId);
        this.createDefaultAnonymizationFields(spaceId);
      }
    } catch (error) {
      this.options.logger.error(`Error initializing AI assistant namespace level resources: ${error.message}`);
      throw error;
    }
  }
  async createDefaultAnonymizationFields(spaceId) {
    const dataClient = new _ai_assistant_data_clients.AIAssistantDataClient({
      logger: this.options.logger,
      elasticsearchClientPromise: this.options.elasticsearchClientPromise,
      spaceId,
      kibanaVersion: this.options.kibanaVersion,
      indexPatternsResorceName: this.resourceNames.aliases.anonymizationFields,
      currentUser: null
    });
    const existingAnonymizationFields = await (await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.getReader())).search({
      body: {
        size: 1
      },
      allow_no_indices: true
    });
    if (existingAnonymizationFields.hits.total.value === 0) {
      const writer = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.getWriter());
      const res = await (writer === null || writer === void 0 ? void 0 : writer.bulk({
        documentsToCreate: (0, _anonymization.getDefaultAnonymizationFields)(spaceId)
      }));
      this.options.logger.info(`Created default anonymization fields: ${res === null || res === void 0 ? void 0 : res.docs_created.length}`);
    }
  }
}
exports.AIAssistantService = AIAssistantService;