"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPureDatasetsSummaryPanelStateMachine = exports.createDatasetsSummaryPanelStateMachine = void 0;
var _timerange = require("@kbn/timerange");
var _xstate = require("xstate");
var _constants = require("../../../../common/constants");
var _filter_inactive_datasets = require("../../../utils/filter_inactive_datasets");
var _defaults = require("./defaults");
var _notifications = require("./notifications");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPureDatasetsSummaryPanelStateMachine = initialContext => /** @xstate-layout N4IgpgJg5mDOIC5QGUCuBbdBDATgTwAUsA7MAGwDoIsAXLWMG2ARVSzIEsa8KAzRgMYALDsSgBiCAHtSFUQDcpAazAU0mXIRLkqteoxZtO3PoJFiECqQNocZAbQAMAXSfPEoAA5TYXO8Q8QAA9EABZQgCYKAHYAVgBmADZQ6IAOUNjQgE5YgEZYgBoQPEQAWlTcilyUx1q6lNSI0IBfZqL1bHwiUkpqOgYmVnYuHn4aYVEJMBwcKRwKTzJaXjn0NQxOrR7dfoMh41GzSctiRRsafzc3QO9fC5lAkIRI1IoK3IrE1LzonMSikoICJZChZL5NUKOTJfZJZFptEAdTTdHR9fSDIwjUzjcxTGZzBZLGgrHBrJFdbS9PQDQzDExjCYWKznS4uey5dxIEC3PwPLlPSKxCjAxJNRzxVKJLJw1IAxAfUIxdK5XLRaK5Rzg2KpVrtDbIyk7dGwACCAgu8ixDNxkhkqisKnWGgp2zRNLNFqtRyZp2stgcbJcNx8vIC-LCsSi2u+EVyEXiWUc0SlhWKiATlWqwNjwNSqSy8VisV1iP1LtR1IMHo4lvp3rxs3mi2WqydmxRVN2TGrtcOOOOzP9xCuQa5PPuYdAAviitCqWi4oSsUThdCcueWUqsca0ay0Xi+9CiRL5K2Fa7pvNNa9-bE4mmjcJLdJbYNrsr3avvexjKgJzOQ5XBywZ3P4jxhDOFBzguhaFiumTrokCRKtUjhgrk0qNNEJ5lmelBwBc2A0JAAAieg-ja0iyA6qinh2FAERwRGkeR1oDr6LIBq4o5eCGE7gc8hYULE0SRAejh5jkcLrpKiQUIk8QSbkimOLkSHxLkOHOnhDGwIRtAsXQFGTPe+JNkSJJkrh9GMcxEBkUZbE+gBE4jpyvGgXyU5hEm8kSvm8SKTBXzrlk+ZVKJEpgpKmTAlp7aGrZBn2ax9amY+zbEq2dGJXpTHJQ5WDGc5fquWywFjnxYHhs8jiKnG8QRKkdXpPGM7riqdWghh4SSrUcQZK0CLEFIEBwIEOU9CBoYCQuMQJI4EQRNEESxOKkTROupSrUKvxNcCjVQgWXzxW+57GvsIzTfxNVxlEFQ-JqMLigpW1qa8R7SotETfeqKSneWnYXZida3lA13Vd5CCqo4bz5DDiTPUk8TrnkwlFsuiYLgWinpADOlunsIM8DgjD4JMENecEiCI7DSFhdqSTLYkSYdQeFCBb8qpZE1PORvCerafRhMYnSPBkFIWBjRAlOTtTCBpJmy0ibG8MY4hsNQtkR5QvkB348LH6Xp63CywJKogtqLNLZkjSJpK66KfEcPShESGQk0arHgik3ne6X43r+Zu3Yk0TCX1NtzsCEn-GmG5yd8cbqhJc5wd7gsJe+F49lipM0OTYjB1DofO3kGluy8GShCjcfLnJyS2wu8aM1kBuGiLxvXiYEtS5ARfy2kiqFqHPM5NU+6x4Ca1CoWmqxizi0Hq3PvWbl+nESldD908KqvHEilLSta0zst65qiCQWJmFvxFqtbfbElG+FcV4OVZ5cs74j80H8tq3rafcc9xCjnIjJCzNGqBXvjoR+hkip5wLq-DyM0aorRBC9cIa1q7JmUh1I63V3r5Fkn9KB+E8p2Wfj3aW29EB2zhitZMjUmiBVlIA-cbxQhxm1GhcIqoIhDWaEAA */
(0, _xstate.createMachine)({
  context: initialContext,
  predictableActionArguments: true,
  id: 'DatasetsQualitySummaryPanel',
  type: 'parallel',
  states: {
    datasetsQuality: {
      initial: 'fetching',
      states: {
        fetching: {
          invoke: {
            src: 'loadDatasetsQuality',
            onDone: {
              target: 'loaded',
              actions: ['storeDatasetsQuality']
            },
            onError: [{
              target: 'retrying',
              cond: {
                type: 'canRetry',
                counter: 'datasetsQualityRetries'
              },
              actions: ['incrementDatasetsQualityRetries']
            }, {
              target: 'loaded',
              actions: ['notifyFetchDatasetsQualityFailed']
            }]
          }
        },
        retrying: {
          after: {
            [_defaults.RETRY_DELAY_IN_MS]: 'fetching'
          }
        },
        loaded: {
          type: 'final'
        }
      }
    },
    datasetsActivity: {
      initial: 'fetching',
      states: {
        fetching: {
          invoke: {
            src: 'loadDatasetsActivity',
            onDone: {
              target: 'loaded',
              actions: ['storeDatasetsActivity']
            },
            onError: [{
              target: 'retrying',
              cond: {
                type: 'canRetry',
                counter: 'datasetsActivityRetries'
              },
              actions: ['incrementDatasetsActivityRetries']
            }, {
              target: 'loaded',
              actions: ['notifyFetchDatasetsActivityFailed']
            }]
          }
        },
        retrying: {
          after: {
            [_defaults.RETRY_DELAY_IN_MS]: 'fetching'
          }
        },
        loaded: {
          type: 'final'
        }
      }
    },
    estimatedData: {
      initial: 'fetching',
      states: {
        fetching: {
          invoke: {
            src: 'loadEstimatedData',
            onDone: [{
              target: 'disabled',
              cond: {
                type: 'estimatedDataIsDisabled'
              }
            }, {
              target: 'loaded',
              actions: ['storeEstimatedData']
            }],
            onError: [{
              target: 'retrying',
              cond: {
                type: 'canRetry',
                counter: 'estimatedDataRetries'
              },
              actions: ['incrementEstimatedDataRetries']
            }, {
              target: 'loaded',
              actions: ['notifyFetchEstimatedDataFailed']
            }]
          }
        },
        retrying: {
          after: {
            [_defaults.RETRY_DELAY_IN_MS]: 'fetching'
          }
        },
        loaded: {
          type: 'final'
        },
        disabled: {
          type: 'final'
        }
      }
    }
  }
}, {
  actions: {
    storeDatasetsQuality: (0, _xstate.assign)((_context, event) => 'data' in event ? {
      datasetsQuality: event.data
    } : {}),
    storeDatasetsActivity: (0, _xstate.assign)((_context, event) => 'data' in event ? {
      datasetsActivity: event.data
    } : {}),
    storeEstimatedData: (0, _xstate.assign)((_context, event) => 'data' in event ? {
      estimatedData: event.data
    } : {}),
    incrementDatasetsQualityRetries: (0, _xstate.assign)(({
      retries
    }, _event) => ({
      retries: {
        ...retries,
        datasetsQualityRetries: retries.datasetsQualityRetries + 1
      }
    })),
    incrementDatasetsActivityRetries: (0, _xstate.assign)(({
      retries
    }, _event) => ({
      retries: {
        ...retries,
        datasetsActivityRetries: retries.datasetsActivityRetries + 1
      }
    })),
    incrementEstimatedDataRetries: (0, _xstate.assign)(({
      retries
    }, _event) => ({
      retries: {
        ...retries,
        estimatedDataRetries: retries.estimatedDataRetries + 1
      }
    }))
  },
  guards: {
    canRetry: (context, event, {
      cond
    }) => {
      if ('counter' in cond && cond.counter in context.retries) {
        const retriesKey = cond.counter;
        return context.retries[retriesKey] < _defaults.MAX_RETRIES;
      }
      return false;
    },
    estimatedDataIsDisabled: (context, event) => {
      return 'estimatedDataInBytes' in event.data && event.data.estimatedDataInBytes === null;
    }
  }
});
exports.createPureDatasetsSummaryPanelStateMachine = createPureDatasetsSummaryPanelStateMachine;
const createDatasetsSummaryPanelStateMachine = ({
  initialContext = _defaults.defaultContext,
  toasts,
  dataStreamStatsClient
}) => createPureDatasetsSummaryPanelStateMachine(initialContext).withConfig({
  actions: {
    notifyFetchDatasetsQualityFailed: (_context, event) => (0, _notifications.fetchDatasetsQualityFailedNotifier)(toasts, event.data),
    notifyFetchDatasetsActivityFailed: (_context, event) => (0, _notifications.fetchDatasetsActivityFailedNotifier)(toasts, event.data),
    notifyFetchEstimatedDataFailed: (_context, event) => (0, _notifications.fetchDatasetsEstimatedDataFailedNotifier)(toasts, event.data)
  },
  services: {
    loadDatasetsQuality: async _context => {
      const dataStreamsStats = await dataStreamStatsClient.getDataStreamsDegradedStats();
      const percentages = dataStreamsStats.map(stat => stat.percentage);
      return {
        percentages
      };
    },
    loadDatasetsActivity: async _context => {
      const dataStreamStats = await dataStreamStatsClient.getDataStreamsStats();
      const activeDataStreams = (0, _filter_inactive_datasets.filterInactiveDatasets)({
        datasets: dataStreamStats
      });
      return {
        total: dataStreamStats.length,
        active: activeDataStreams.length
      };
    },
    loadEstimatedData: async _context => {
      const {
        startDate,
        endDate
      } = (0, _timerange.getDateISORange)(_constants.DEFAULT_TIME_RANGE);
      return dataStreamStatsClient.getDataStreamsEstimatedDataInBytes({
        query: {
          type: 'logs',
          start: startDate,
          end: endDate
        }
      });
    }
  }
});
exports.createDatasetsSummaryPanelStateMachine = createDatasetsSummaryPanelStateMachine;