"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDatasetQualityFilters = void 0;
var _react = require("@xstate/react");
var _react2 = require("react");
var _integration = require("../../common/data_streams_stats/integration");
var _context = require("../components/dataset_quality/context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useDatasetQualityFilters = () => {
  const {
    service
  } = (0, _context.useDatasetQualityContext)();
  const isLoading = (0, _react.useSelector)(service, state => state.matches('integrations.fetching') && (state.matches('datasets.fetching') || state.matches('degradedDocs.fetching')));
  const {
    timeRange,
    integrations: selectedIntegrations,
    namespaces: selectedNamespaces,
    query: selectedQuery
  } = (0, _react.useSelector)(service, state => state.context.filters);
  const datasets = (0, _react.useSelector)(service, state => state.context.datasets);
  const namespaces = (0, _react.useSelector)(service, state => state.context.datasets).map(dataset => dataset.namespace);
  const onTimeChange = (0, _react2.useCallback)(selectedTime => {
    if (selectedTime.isInvalid) {
      return;
    }
    service.send({
      type: 'UPDATE_TIME_RANGE',
      timeRange: {
        ...timeRange,
        from: selectedTime.start,
        to: selectedTime.end
      }
    });
  }, [service, timeRange]);
  const onRefresh = (0, _react2.useCallback)(() => {
    service.send({
      type: 'REFRESH_DATA'
    });
  }, [service]);
  const onRefreshChange = (0, _react2.useCallback)(({
    refreshInterval,
    isPaused
  }) => {
    service.send({
      type: 'UPDATE_TIME_RANGE',
      timeRange: {
        ...timeRange,
        refresh: {
          pause: isPaused,
          value: refreshInterval
        }
      }
    });
  }, [service, timeRange]);
  const integrationItems = (0, _react2.useMemo)(() => {
    const integrations = [...datasets.map(dataset => dataset.integration).filter(integration => !!integration), ...(datasets.some(dataset => !dataset.integration) ? [_integration.Integration.create({
      name: 'none',
      title: 'None'
    })] : [])];
    return integrations.map(integration => ({
      ...integration,
      label: integration.title,
      checked: selectedIntegrations.includes(integration.name) ? 'on' : undefined
    }));
  }, [datasets, selectedIntegrations]);
  const onIntegrationsChange = (0, _react2.useCallback)(newIntegrationItems => {
    service.send({
      type: 'UPDATE_INTEGRATIONS',
      integrations: newIntegrationItems.filter(integration => integration.checked === 'on').map(integration => integration.name)
    });
  }, [service]);
  const namespaceItems = (0, _react2.useMemo)(() => {
    const uniqueNamespaces = [...new Set(namespaces)];
    return uniqueNamespaces.map(namespace => ({
      label: namespace,
      checked: selectedNamespaces.includes(namespace) ? 'on' : undefined
    }));
  }, [namespaces, selectedNamespaces]);
  const onNamespacesChange = (0, _react2.useCallback)(newNamespaceItems => {
    service.send({
      type: 'UPDATE_NAMESPACES',
      namespaces: newNamespaceItems.filter(namespace => namespace.checked === 'on').map(namespace => namespace.label)
    });
  }, [service]);
  const onQueryChange = (0, _react2.useCallback)(query => {
    service.send({
      type: 'UPDATE_QUERY',
      query
    });
  }, [service]);
  return {
    timeRange,
    onTimeChange,
    onRefresh,
    onRefreshChange,
    integrations: integrationItems,
    namespaces: namespaceItems,
    onIntegrationsChange,
    onNamespacesChange,
    isLoading,
    selectedQuery,
    onQueryChange
  };
};
exports.useDatasetQualityFilters = useDatasetQualityFilters;