"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useESQLFieldStatsData = void 0;
var _i18n = require("@kbn/i18n");
var _react = require("react");
var _lodash = require("lodash");
var _mlCancellableSearch = require("@kbn/ml-cancellable-search");
var _esqlUtils = require("@kbn/esql-utils");
var _kibana_context = require("../../../kibana_context");
var _progress_utils = require("../../progress_utils");
var _esql_utils = require("../../search_strategy/requests/esql_utils");
var _get_numeric_field_stats = require("../../search_strategy/esql_requests/get_numeric_field_stats");
var _get_keyword_fields = require("../../search_strategy/esql_requests/get_keyword_fields");
var _get_date_field_stats = require("../../search_strategy/esql_requests/get_date_field_stats");
var _get_boolean_field_stats = require("../../search_strategy/esql_requests/get_boolean_field_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useESQLFieldStatsData = ({
  searchQuery,
  columns: allColumns,
  filter,
  limit
}) => {
  const [fieldStats, setFieldStats] = (0, _react.useState)();
  const [fetchState, setFetchState] = (0, _react.useReducer)((0, _progress_utils.getReducer)(), (0, _progress_utils.getInitialProgress)());
  const {
    services: {
      data,
      notifications: {
        toasts
      }
    }
  } = (0, _kibana_context.useDataVisualizerKibana)();
  const {
    runRequest,
    cancelRequest
  } = (0, _mlCancellableSearch.useCancellableSearch)(data);
  (0, _react.useEffect)(function updateFieldStats() {
    let unmounted = false;
    const fetchFieldStats = async () => {
      cancelRequest();
      setFetchState({
        ...(0, _progress_utils.getInitialProgress)(),
        isRunning: true,
        error: undefined
      });
      if (!(0, _esql_utils.isESQLQuery)(searchQuery) || !allColumns) return;
      try {
        // By default, limit the source data to 100,000 rows
        const esqlBaseQuery = (0, _esqlUtils.getESQLWithSafeLimit)(searchQuery.esql, limit);
        const totalFieldsCnt = allColumns.length;
        const processedFieldStats = new Map();
        function addToProcessedFieldStats(stats) {
          if (!unmounted) {
            stats.forEach(field => {
              if (field) {
                processedFieldStats.set(field.fieldName, field);
              }
            });
            setFetchState({
              loaded: processedFieldStats.size / totalFieldsCnt * 100
            });
          }
        }
        setFieldStats(processedFieldStats);
        const aggregatableFieldsChunks = (0, _lodash.chunk)(allColumns, 25);
        for (const columns of aggregatableFieldsChunks) {
          // GETTING STATS FOR NUMERIC FIELDS
          await (0, _get_numeric_field_stats.getESQLNumericFieldStats)({
            columns: columns.filter(f => f.secondaryType === 'number'),
            filter,
            runRequest,
            esqlBaseQuery
          }).then(addToProcessedFieldStats);

          // GETTING STATS FOR KEYWORD FIELDS
          await (0, _get_keyword_fields.getESQLKeywordFieldStats)({
            columns: columns.filter(f => f.secondaryType === 'keyword' || f.secondaryType === 'ip'),
            filter,
            runRequest,
            esqlBaseQuery
          }).then(addToProcessedFieldStats);

          // GETTING STATS FOR BOOLEAN FIELDS
          await (0, _get_boolean_field_stats.getESQLBooleanFieldStats)({
            columns: columns.filter(f => f.secondaryType === 'boolean'),
            filter,
            runRequest,
            esqlBaseQuery
          }).then(addToProcessedFieldStats);

          // GETTING STATS FOR DATE FIELDS
          await (0, _get_date_field_stats.getESQLDateFieldStats)({
            columns: columns.filter(f => f.secondaryType === 'date'),
            filter,
            runRequest,
            esqlBaseQuery
          }).then(addToProcessedFieldStats);
        }
        setFetchState({
          loaded: 100,
          isRunning: false
        });
      } catch (e) {
        if (e.name !== 'AbortError') {
          const title = _i18n.i18n.translate('xpack.dataVisualizer.index.errorFetchingESQLFieldStatisticsMessage', {
            defaultMessage: 'Error fetching field statistics for ES|QL query'
          });
          toasts.addError(e, {
            title
          });

          // Log error to console for better debugging
          // eslint-disable-next-line no-console
          console.error(`${title}: fetchFieldStats`, e);
          setFetchState({
            loaded: 100,
            isRunning: false,
            error: e
          });
        }
      }
    };
    fetchFieldStats();
    return () => {
      unmounted = true;
    };
  },
  // eslint-disable-next-line react-hooks/exhaustive-deps
  [allColumns, JSON.stringify({
    filter
  }), limit]);
  return {
    fieldStats,
    fieldStatsProgress: fetchState,
    cancelFieldStatsRequest: cancelRequest
  };
};
exports.useESQLFieldStatsData = useESQLFieldStatsData;