"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FieldEditorFlyoutContentContainer = void 0;
var _react = _interopRequireWildcard(require("react"));
var _i18n = require("@kbn/i18n");
var _analytics = require("@kbn/analytics");
var _rxjs = require("rxjs");
var _constants = require("../constants");
var _lib = require("../lib");
var _field_editor_flyout_content = require("./field_editor_flyout_content");
var _field_editor_context = require("./field_editor_context");
var _preview = require("./preview");
var _preview_controller = require("./preview/preview_controller");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * The container component will be in charge of the communication with the index pattern service
 * to retrieve/save the field in the saved object.
 * The <FieldEditorFlyoutContent /> component is the presentational component that won't know
 * anything about where a field comes from and where it should be persisted.
 */

const FieldEditorFlyoutContentContainer = ({
  fieldToEdit,
  fieldToCreate,
  onSave,
  onCancel,
  onMounted,
  docLinks,
  fieldTypeToProcess,
  dataView,
  dataViews,
  search,
  notifications,
  usageCollection,
  apiService,
  fieldFormatEditors,
  fieldFormats,
  uiSettings
}) => {
  const [controller] = (0, _react.useState)(() => new _preview_controller.PreviewController({
    dataView,
    search,
    fieldFormats,
    fieldTypeToProcess
  }));
  const [isSaving, setIsSaving] = (0, _react.useState)(false);
  const {
    fields
  } = dataView;
  const namesNotAllowed = (0, _react.useMemo)(() => {
    const fieldNames = dataView.fields.map(fld => fld.name);
    const runtimeCompositeNames = Object.entries(dataView.getAllRuntimeFields()).filter(([, _runtimeField]) => _runtimeField.type === 'composite').map(([_runtimeFieldName]) => _runtimeFieldName);
    return {
      fields: fieldNames,
      runtimeComposites: runtimeCompositeNames
    };
  }, [dataView]);
  const existingConcreteFields = (0, _react.useMemo)(() => {
    const existing = [];
    fields.filter(fld => {
      const isFieldBeingEdited = (fieldToEdit === null || fieldToEdit === void 0 ? void 0 : fieldToEdit.name) === fld.name;
      return !isFieldBeingEdited && fld.isMapped;
    }).forEach(fld => {
      existing.push({
        name: fld.name,
        type: fld.esTypes && fld.esTypes[0] || ''
      });
    });
    return existing;
  }, [fields, fieldToEdit]);
  const services = (0, _react.useMemo)(() => ({
    api: apiService,
    search,
    notifications
  }), [apiService, search, notifications]);
  const updateRuntimeField = (0, _react.useCallback)(updatedField => {
    const nameHasChanged = Boolean(fieldToEdit) && fieldToEdit.name !== updatedField.name;
    const typeHasChanged = Boolean(fieldToEdit) && fieldToEdit.type !== updatedField.type;
    const hasChangeToOrFromComposite = typeHasChanged && (fieldToEdit.type === 'composite' || updatedField.type === 'composite');
    const {
      script
    } = updatedField;
    if (fieldTypeToProcess === 'runtime') {
      try {
        usageCollection.reportUiCounter(_constants.pluginName, _analytics.METRIC_TYPE.COUNT, 'save_runtime');
        // eslint-disable-next-line no-empty
      } catch {}
      // rename an existing runtime field
      if (nameHasChanged || hasChangeToOrFromComposite) {
        dataView.removeRuntimeField(fieldToEdit.name);
      }
      dataView.addRuntimeField(updatedField.name, {
        type: updatedField.type,
        script,
        fields: updatedField.fields
      });
    } else {
      try {
        usageCollection.reportUiCounter(_constants.pluginName, _analytics.METRIC_TYPE.COUNT, 'save_concrete');
        // eslint-disable-next-line no-empty
      } catch {}
    }
    return dataView.addRuntimeField(updatedField.name, updatedField);
  }, [fieldToEdit, dataView, fieldTypeToProcess, usageCollection]);
  const updateConcreteField = (0, _react.useCallback)(updatedField => {
    const editedField = dataView.getFieldByName(updatedField.name);
    if (!editedField) {
      throw new Error(`Unable to find field named '${updatedField.name}' on index pattern '${dataView.getIndexPattern()}'`);
    }

    // Update custom label, popularity and format
    dataView.setFieldCustomLabel(updatedField.name, updatedField.customLabel);
    dataView.setFieldCustomDescription(updatedField.name, updatedField.customDescription);
    editedField.count = updatedField.popularity || 0;
    if (updatedField.format) {
      dataView.setFieldFormat(updatedField.name, updatedField.format);
    } else {
      dataView.deleteFieldFormat(updatedField.name);
    }
    return [editedField];
  }, [dataView]);
  const saveField = (0, _react.useCallback)(async updatedField => {
    try {
      usageCollection.reportUiCounter(_constants.pluginName, _analytics.METRIC_TYPE.COUNT, fieldTypeToProcess === 'runtime' ? 'save_runtime' : 'save_concrete');
      // eslint-disable-next-line no-empty
    } catch {}
    setIsSaving(true);
    try {
      const editedFields = fieldTypeToProcess === 'runtime' ? updateRuntimeField(updatedField) : updateConcreteField(updatedField);
      const afterSave = () => {
        const message = _i18n.i18n.translate('indexPatternFieldEditor.deleteField.savedHeader', {
          defaultMessage: "Saved '{fieldName}'",
          values: {
            fieldName: updatedField.name
          }
        });
        notifications.toasts.addSuccess(message);
        setIsSaving(false);
        onSave(editedFields);
      };
      if (dataView.isPersisted()) {
        await dataViews.updateSavedObject(dataView);
      }
      afterSave();
      setIsSaving(false);
    } catch (e) {
      const title = _i18n.i18n.translate('indexPatternFieldEditor.save.errorTitle', {
        defaultMessage: 'Failed to save field changes'
      });
      notifications.toasts.addError(e, {
        title
      });
      setIsSaving(false);
    }
  }, [onSave, dataView, dataViews, notifications, fieldTypeToProcess, updateConcreteField, updateRuntimeField, usageCollection]);
  return /*#__PURE__*/_react.default.createElement(_field_editor_context.FieldEditorProvider, {
    dataView: dataView,
    uiSettings: uiSettings,
    links: (0, _lib.getLinks)(docLinks),
    fieldTypeToProcess: fieldTypeToProcess,
    services: services,
    fieldFormatEditors: fieldFormatEditors,
    fieldFormats: fieldFormats,
    namesNotAllowed: namesNotAllowed,
    existingConcreteFields: existingConcreteFields,
    fieldName$: new _rxjs.BehaviorSubject((fieldToEdit === null || fieldToEdit === void 0 ? void 0 : fieldToEdit.name) || ''),
    subfields$: new _rxjs.BehaviorSubject(fieldToEdit === null || fieldToEdit === void 0 ? void 0 : fieldToEdit.fields)
  }, /*#__PURE__*/_react.default.createElement(_preview.FieldPreviewProvider, {
    controller: controller
  }, /*#__PURE__*/_react.default.createElement(_field_editor_flyout_content.FieldEditorFlyoutContent, {
    onSave: saveField,
    onCancel: onCancel,
    onMounted: onMounted,
    fieldToCreate: fieldToCreate,
    fieldToEdit: fieldToEdit,
    isSavingField: isSaving
  })));
};
exports.FieldEditorFlyoutContentContainer = FieldEditorFlyoutContentContainer;