"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.eqlSearchStrategyProvider = void 0;
var _rxjs = require("rxjs");
var _server = require("@kbn/kibana-utils-plugin/server");
var _common = require("../../../../common");
var _response_utils = require("./response_utils");
var _es_search = require("../es_search");
var _request_utils = require("../ese_search/request_utils");
var _async_utils = require("../common/async_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const eqlSearchStrategyProvider = (searchConfig, logger) => {
  function cancelAsyncSearch(id, esClient) {
    const client = esClient.asCurrentUser.eql;
    return client.delete({
      id
    });
  }
  return {
    cancel: async (id, options, {
      esClient
    }) => {
      logger.debug(`_eql/delete ${id}`);
      try {
        await cancelAsyncSearch(id, esClient);
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    },
    search: ({
      id,
      ...request
    }, options, {
      esClient,
      uiSettingsClient
    }) => {
      logger.debug(`_eql/search ${JSON.stringify(request.params) || id}`);
      const client = esClient.asCurrentUser.eql;
      const search = async () => {
        var _meta, _meta$request;
        const {
          track_total_hits: _,
          ...defaultParams
        } = await (0, _es_search.getDefaultSearchParams)(uiSettingsClient);
        const params = id ? (0, _async_utils.getCommonDefaultAsyncGetParams)(searchConfig, options, {
          /* disable until full eql support */disableSearchSessions: true
        }) : {
          ...(await (0, _request_utils.getIgnoreThrottled)(uiSettingsClient)),
          ...defaultParams,
          ...(0, _async_utils.getCommonDefaultAsyncGetParams)(searchConfig, options, {
            /* disable until full eql support */disableSearchSessions: true
          }),
          ...request.params
        };
        const response = id ? await client.get({
          ...params,
          id
        }, {
          ...request.options,
          ...options.transport,
          signal: options.abortSignal,
          meta: true
        }) :
        // @ts-expect-error optional key cannot be used since search doesn't expect undefined
        await client.search(params, {
          ...request.options,
          ...options.transport,
          signal: options.abortSignal,
          meta: true
        });
        return (0, _response_utils.toEqlKibanaSearchResponse)(response,
        // do not return requestParams on polling calls
        id ? undefined : (_meta = response.meta) === null || _meta === void 0 ? void 0 : (_meta$request = _meta.request) === null || _meta$request === void 0 ? void 0 : _meta$request.params);
      };
      const cancel = async () => {
        if (!id) return;
        try {
          await cancelAsyncSearch(id, esClient);
        } catch (e) {
          var _e$meta;
          // A 404 means either this search request does not exist, or that it is already cancelled
          if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) === 404) return;

          // Log all other (unexpected) error messages
          logger.error(`cancelEqlSearch error: ${e.message}`);
        }
      };
      return (0, _common.pollSearch)(search, cancel, {
        pollInterval: searchConfig.asyncSearch.pollInterval,
        ...options
      }).pipe((0, _rxjs.tap)(response => id = response.id));
    },
    extend: async (id, keepAlive, options, {
      esClient
    }) => {
      logger.debug(`_eql/extend ${id} by ${keepAlive}`);
      await esClient.asCurrentUser.eql.get({
        id,
        keep_alive: keepAlive
      });
    }
  };
};
exports.eqlSearchStrategyProvider = eqlSearchStrategyProvider;